<?php
namespace app\index\controller;

use app\common\controller\FrontController;
use app\common\model\Trade as TradeModel;
use app\common\model\TradeCollect;
use app\common\model\TradeGold;
use app\common\model\TradeProp;
use app\common\model\TradeRole;
use think\App;

class Trade extends FrontController
{
    /**
     *  类型： 0-金币 1-角色 2-道具
     * @var int
     */
    protected $type = TradeModel::TYPE_ROLES;

    /**
     *  筛选 1-浏览商品 2-公示期商品 3-指定我
     * @var int
     */
    protected $mode = 1;

    protected $templateLists = [
        TradeModel::TYPE_GOLD => "trade/gold",
        TradeModel::TYPE_ROLES => "trade/role",
        TradeModel::TYPE_PROP => "trade/prop",
    ];

    public function __construct(App $app)
    {
        parent::__construct($app);
        $params = $this->request->param();
        $this->type = $params['type'] ?? 1;
        $this->mode = $params['mode'] ?? 1;

        $this->assign([
            'type' => TradeModel::TYPE[$this->type] ?? "",
            'mode' => $this->mode,
            'params' => $params,
            'query' => http_build_query($params),
        ]);
    }

    /**
     *  列表首页
     * @return string
     */
    public function index()
    {
        $d = $this->request->get();

        $where = [
            'game'     => $this->game,
            'serverid' => $this->serverid,
            'pay_status' => TradeModel::TYPE_PAY_STATUS_NO,
            ['trade_status', 'IN', [TradeModel::T_TRADE_STATUS_PUB, TradeModel::T_TRADE_STATUS_SELL]]
        ];

        // 排序
        $key   = $d['key'] ?? '';
        $value = $d['value'] ?? '';
        $order = ['id' => 'desc'];
        if (in_array($key, ['unit_price', 'price', 'outtime']) && $key && $value) {
            $order = [$key => $value];
        }

        switch ($this->mode) {
            case 1: // 浏览商品列表
                $where[] = ["created_at", "BETWEEN", [
                        date("Y-m-d H:i:s", (time() - (int)bcadd(config('app.trade_seller_time'), config('app.trade_public_time')))),
                        date("Y-m-d H:i:s", (time() - (int)config('app.trade_public_time'))),
                    ]
                ];
                $where[] = ['specuser', '=', ''];
                break;
            case 2: // 公示期列表
                $where[] = ["created_at", ">=", date("Y-m-d H:i:s", (time() - (int)config('app.trade_public_time')))];
                break;
            case 3: // 指定我列表
                $where[]           = ["created_at", ">=", date("Y-m-d H:i:s", (time() - bcadd(config('app.trade_public_time'), config('app.trade_seller_time'))))];
                $where["specuser"] = $this->request->member['username'] ?? "-1";
                break;
        }

        $model    = TradeModel::getTypeTradeModel($this->type);
        $modelObj = app('app\\common\\model\\' . $model);
        $data     = $modelObj->listIndexQuery($where, $order, 8, [
            'game'     => $this->game,
            'serverid' => $this->serverid
        ]);

        $this->assign([
            'lists' => $data['lists'] ?? [],
            'page'  => $data['page'] ?? "",
        ]);
        return $this->fetch($this->templateLists[$this->type] ?? "/");
    }

    /**
     *  角色详情
     * @return string
     */
    public function role()
    {
        $id     = (int)$this->request->param('id');
        $member = $this->request->member;

        $data = (new TradeRole())->getDetails($id, $this->game, $this->serverid);
        $isCollect = (new TradeCollect())->isCollect($this->game, $this->serverid, $member['username'] ?? "", $id, TradeModel::TYPE_ROLES);

        $this->assign([
            'details'           => $data['details'] ?? [],
            'trade'             => $data['trade'] ?? [],
            'warLists'          => $data['warLists'] ?? [],
            'equipLists'        => $data['equipLists'] ?? [],
            'stLists'           => $data['stLists'] ?? [],
            'packsackLists'     => $data['packsackLists'] ?? [], // 背包道具
            'roleBackpackLists' => $data['roleBackpackLists'] ?? [], // 角色仓库
            'facadeLists'       => $data['facadeLists'] ?? [], // 外观
            'godPartLists'      => $data['godPartLists'] ?? [], // 天神属性列表
//            'godTotalLists'     => $data['godTotalLists'] ?? [], // 天神总属性
            'wingLists'         => $data['wingLists'] ?? [], // 翅膀属性列表
//            'wingTotalLists'    => $data['wingTotalLists'] ?? [], // 翅膀总属性
            'itemTypeLists'     => config($this->game . ".item_type_lists"), // 分类
            'roleItemTips'      => config($this->game . ".role_item_details_tips"),
            'is_collect'        => $isCollect,
        ]);
        return $this->fetch("trade/role_details");
    }
    /**
     *  金币详情
     * @return string
     */
    public function gold()
    {
        $id     = (int)$this->request->param('id');
        $member = $this->request->member;

        $data = (new TradeGold())->getDetails($id, $this->game, $this->serverid);
        $isCollect = (new TradeCollect())->isCollect($this->game, $this->serverid, $member['username'] ?? "", $id, TradeModel::TYPE_GOLD);

        $this->assign([
            'details'    => $data['details'] ?? [],
            'is_collect' => $isCollect,
        ]);
        return $this->fetch("trade/gold_details");
    }
    /**
     *  道具详情
     * @return string
     */
    public function prop()
    {
        $id     = (int)$this->request->param('id');
        $member = $this->request->member;

        $data = (new TradeProp())->getDetails($id, $this->game, $this->serverid);
        $isCollect = (new TradeCollect())->isCollect($this->game, $this->serverid, $member['username'] ?? "", $id, TradeModel::TYPE_PROP);

        $this->assign([
            'details'    => $data['details'] ?? [],
            'item'       => $data['item'] ?? [],
            'is_collect' => $isCollect,
        ]);
        return $this->fetch("trade/prop_details");
    }

    /**
     *  收藏
     * @return \think\response\Json
     * @throws \think\db\exception\DataNotFoundException
     * @throws \think\db\exception\DbException
     * @throws \think\db\exception\ModelNotFoundException
     */
    public function collect()
    {
        $id = (int)$this->request->param('id');
        if (!$id) {
            return $this->fail("缺少参数");
        }

        $type = (int)$this->request->param('type');
        if (!in_array($type, array_keys(TradeModel::TYPE))) {
            return $this->fail("缺少参数");
        }

        $username = $this->request->member['username'] ?? 0;

        $model    = TradeModel::getTypeTradeModel($this->type);
        $modelObj = app('app\\common\\model\\' . $model);
        $checkRes = $modelObj->getDetailsByWhere(['id' => $id]);
        if (!$checkRes) {
            return $this->fail("交易不存在");
        }

        $details = TradeCollect::where([
            "username"   => $username,
            "tid"        => $checkRes['tid'],
            "type"       => $type,
            'game'       => $this->game,
            'serverid'   => $this->serverid,
            "relationid" => $id,
        ])->find();
        if ($details) {
            $res = TradeCollect::where(['id' => $details->id])->delete();
        } else {
            $res = TradeCollect::insertGetId([
                "username"   => $username,
                "tid"        => $checkRes['tid'],
                "type"       => $type,
                "game"       => $this->game,
                "serverid"   => $this->serverid,
                "relationid" => $id,
                "price"      => $checkRes['price'] ?? 0,
                "outtime"    => $checkRes['outtime'] ?? 0,
            ]);
        }
        if (false === $res) {
            return $this->fail("操作失败");
        }

        return $this->success("操作成功", ['is_collect' => (bool)$details]);
    }
}