<?php

namespace app\index\controller;

use app\common\constant\CacheKey;
use app\common\constant\CommonKey;
use app\common\controller\FrontController;
use app\common\model\Trade;
use app\common\model\Trade as TradeModel;
use app\common\model\TradeEquip;
use app\common\model\TradeItems;
use app\common\model\TradeNotice;
use app\common\model\TradeOrder;
use app\common\service\FileLockService;
use think\App;
use think\Exception;
use think\facade\Db;
use think\facade\Log;

class Seller extends FrontController
{
    public function __construct(App $app)
    {
        parent::__construct($app);
        $params     = $this->request->param();
        $this->mode = $params['mode'] ?? 1;
        $this->assign([
            'type'   => TradeModel::TYPE[$params['type'] ?? 1] ?? "",
            'mode'   => $params['mode'] ?? 1,
            'params' => $params,
            'query'  => http_build_query($params),
        ]);
    }

    /**
     * 列表首页
     * @return string
     */
    public function index()
    {
        $data = [
            'game'     => $this->game,
            'serverid' => $this->serverid,
            'username' => $this->request->member['username'] ?? ""
        ];
        $data = (new TradeModel())->listIndexQuery($data);

        $this->assign([
            'lists' => $data['lists'] ?? [],
            'page'  => $data['page'] ?? "",
        ]);
        return $this->fetch("seller/index");
    }

    /**
     * 详情
     * @return string
     */
    public function details()
    {
        $type = $this->request->param('type');
        $id   = $this->request->param('id');
        $username  = $this->request->member['username'] ?? "";

        $model    = TradeModel::getTypeTradeModel($type);
        $modelObj = app('app\\common\\model\\' . $model);
        $details  = $modelObj->getDetailsByWhere(['id' => $id, 'username' => $username]);
        $item     = [];
        if ($type == TradeModel::TYPE_PROP) {
            $itemid = $details['itemid'] ?? 0;
            $item   = (new TradeItems())->getDetailsByItemid($this->game, $this->serverid, $itemid);

            // 获取职业
            $cjLists = (new TradeEquip())->getEquipListsByTids($this->game, $this->serverid, [$details['tid'] ?? 0], ['cj', 'itemid']);
            $cjLists = array_column($cjLists, "cj", 'itemid');

            $details['cj'] = $cjLists[$itemid] ?? 0;
        }
        $orderInfo = (new TradeOrder())->getOrderInfo($type, $id);
        $details['order_addtime'] = $orderInfo['addtime'] ?? 0;
        $details['order_pay_status'] = $orderInfo['pay_status'] ?? 0;
        $res = $modelObj->handleRawData($details, $item);
        $orderInfo = (new TradeOrder())->handleRawData($orderInfo, $details);

        $this->assign([
            'type'      => $type,
            'details'   => $res['rows'] ?? [],
            'item'      => $res['item'] ?? [],
            'orderinfo' => $orderInfo,
        ]);
        return $this->fetch("seller/details");
    }

    /**
     *  下架
     * @return \think\response\Json
     */
    public function close()
    {
        $type = $this->request->param('type');
        $id   = $this->request->param('id');
        $username = $this->request->member['username'] ?? '';

        try {
            $obj = new FileLockService();
            $lockKey = sprintf(CacheKey::ADMIN_TRADE_CLOSE_LOCK, $type, $id);
            $fp = $obj->acquireLock($lockKey, CommonKey::EXPIRED_THREE_SECOND);
            if ($fp === false) {
                throw new Exception("操作频繁");
            }

            $tradeObj = new TradeModel();
            $model    = TradeModel::getTypeTradeModel($type);
            $modelObj = app('app\\common\\model\\' . $model);
            $details  = $modelObj->getDetailsByWhere(['id' => $id, 'username' => $username], ['id', 'tid', 'trade_status']);
            if (!$details) {
                throw new Exception("该商品不存在");
            }

            $tradeStatus = $details['trade_status'] ?? 0;
            if (in_array($tradeStatus, TradeModel::T_TRADE_STATUS_COLLECT)) {
                throw new Exception("该商品已下架，请勿重复操作");
            }

            if (in_array($tradeStatus, [TradeModel::T_TRADE_STATUS_PLACE, TradeModel::T_TRADE_STATUS_DONE])) {
                throw new Exception("该商品" . (TradeModel::T_TRADE_STATUS[$tradeStatus] ?? "") . "，无法操作");
            }

            $tid         = $details['tid'] ?? 0;
            $trade       = $tradeObj->getDetailsByWhere([
                'id' => $tid, 'game' => $this->game, 'serverid' => $this->serverid,
            ], ["id", "game", "serverid", "username", "type", "sync_status", "sync_tradeno"]);
            $syncTradeno = $trade['sync_tradeno'] ?? "";
            if (!$syncTradeno) {
                throw new Exception("交易数据错误");
            }

            Db::startTrans();
            try {
                // 下架接口
                $tradeObj->closeTrade([
                    'game'         => $this->game,
                    'serverid'     => $this->serverid,
                    'tid'          => $tid,
                    'id'           => $id,
                    'username'     => $username,
                    'type'         => $type,
                    'sync_tradeno' => $syncTradeno,
                    'trade_status' => TradeModel::T_TRADE_STATUS_DOWN,
                    'notice_type'  => TradeNotice::NOTICE_TYPE_OWNER_DOWN,
                    'handle_type'  => Trade::OP_CLOSE_TRADE
                ]);
                Db::commit();
            } catch (\Throwable $e) {
                Db::rollback();
                Log::error("玩家下架-处理出错：" . json_encode([
                        'Message' => $e->getMessage(),
                        'Line'    => $e->getLine(),
                        'File'    => $e->getFile(),
                        'Trace'   => $e->getTraceAsString(),
                    ], JSON_UNESCAPED_UNICODE));
                throw new Exception("下架失败");
            }
        } catch (Exception $th) {
            return $this->fail($th->getMessage());
        } finally {
            if ($fp !== false) {
                $obj->deleteLockFile($fp, $lockKey);
            }
        }

        return $this->success("操作成功");
    }
}