<?php

namespace app\index\controller;

use app\common\constant\CacheKey;
use app\common\constant\CommonKey;
use app\common\controller\FrontController;
use app\common\model\Trade as TradeModel;
use app\common\model\TradeCollect;
use app\common\model\TradeItems;
use app\common\model\TradeOrder;
use app\common\service\FileLockService;
use app\common\service\GameTradeService;
use think\App;
use think\Exception;
use think\facade\Db;
use think\facade\Log;

class My extends FrontController
{
    public function __construct(App $app)
    {
        parent::__construct($app);
        $params     = $this->request->param();
        $this->mode = $params['mode'] ?? 1;
        $this->assign([
            'type'   => TradeModel::TYPE[$params['type'] ?? 1] ?? "",
            'mode'   => $params['mode'] ?? 1,
            'params' => $params,
            'query'  => http_build_query($params),
        ]);
    }

    /**
     *  列表首页
     * @return string
     */
    public function index()
    {
        $data = [
            'game'     => $this->game,
            'serverid' => $this->serverid,
            'username' => $this->request->member['username'] ?? ""
        ];
        $data = (new TradeOrder())->listOrderQuery($data);

        $this->assign([
            'lists' => $data['lists'] ?? [],
            'page'  => $data['page'] ?? "",
        ]);
        return $this->fetch("my/index");
    }

    /**
     *  我的收藏列表
     * @return string
     */
    public function collect()
    {
        $data = [
            'game'     => $this->game,
            'serverid' => $this->serverid,
            'username' => $this->request->member['username'] ?? ""
        ];
        $data = (new TradeCollect())->listCollectQuery($data);

        $this->assign([
            'lists' => $data['lists'] ?? [],
            'page'  => $data['page'] ?? "",
        ]);
        return $this->fetch("my/collect");
    }

    /**
     *  取消订单
     * @return \think\response\Json
     */
    public function cancel()
    {
        if (!$this->request->isPost()) {
            return $this->fail("非法请求");
        }

        $orderno = $this->request->post('orderno');
        if (!$orderno) {
            return $this->fail("缺少参数");
        }

        try {
            $obj = new FileLockService();
            $lockKey = sprintf(CacheKey::INDEX_CANCEL_ORDER_LOCK, $orderno);
            $fp = $obj->acquireLock($lockKey, CommonKey::EXPIRED_THREE_SECOND);
            if ($fp === false) {
                throw new Exception("操作频繁");
            }

            $order = (new TradeOrder())
                ->getDetailsByWhere([
                    'orderno' => $orderno
                ], ['id', 'game', 'serverid', 'ordertype', 'orderno', 'relationid', 'pay_status', 'tid', 'tousername', 'order_details']);
            if (!$order) {
                throw new Exception("订单不存在");
            }

            $paystatus = $order['pay_status'] ?? 0;
            if ($paystatus == TradeOrder::TO_PAY_STATUS_DONE) {
                throw new Exception("订单已支付，无法取消");
            }

            if ($paystatus == TradeOrder::TO_PAY_STATUS_CACEL) {
                throw new Exception("订单已取消");
            }

            Db::startTrans();
            try {
                (new TradeOrder())->cancelOrder($order);
                Db::commit();
            } catch (Exception $e) {
                Db::rollback();
                Log::info("取消订单失败：" . json_encode([
                        'Message' => $e->getMessage(),
                        'Line'    => $e->getLine(),
                        'File'    => $e->getFile(),
                        'Trace'   => $e->getTraceAsString(),
                    ], JSON_UNESCAPED_UNICODE));
                throw new Exception("取消失败");
            }

        } catch (Exception $th) {
            return $this->fail($th->getMessage());
        } finally {
            if ($fp !== false) {
                $obj->deleteLockFile($fp, $lockKey);
            }
        }
        return $this->success("取消成功");
    }

    /**
     * 验证是否创建订单
     */
    public function checkOrder()
    {
        if (!$this->request->isPost()) {
            return $this->fail("非法请求");
        }

        $id   = $this->request->post('id', 0);
        $type = $this->request->post('type', 0);

        try {
            $obj = new FileLockService();
            $lockKey = sprintf(CacheKey::ADMIN_TRADE_CLOSE_LOCK, $type, $id);
            $fp = $obj->acquireLock($lockKey, CommonKey::EXPIRED_THREE_SECOND);
            if ($fp === false) {
                throw new Exception("操作频繁");
            }

            try {
                $tradeInfo = TradeModel::getDynamicTrade($type)->where(['id' => $id])->findOrEmpty();
            } catch (\Exception $e) {
                throw new Exception("商品详情不存在");
            }

            $tradeInfo = $tradeInfo->toArray();
            $model     = TradeModel::getTypeTradeModel($type);
            if (empty($tradeInfo) || !isset($tradeInfo[$model])) {
                throw new Exception("商品不存在");
            }

            $username = $this->request->member['username'] ?? "";
            $mInfo    = $tradeInfo[$model];
            if (!$mInfo) {
                throw new Exception("商品不存在.");
            }

            $tradeStatus = $mInfo['trade_status'] ?? 0;
            // 判断公示中
            if (time() <= (strtotime($mInfo['created_at']) + config('app.trade_public_time'))) {
                throw new Exception("商品公示中，无法购买");
            }
            // 判断是否被下单
            if ($tradeStatus == TradeModel::T_TRADE_STATUS_PLACE) {
                throw new Exception("商品已被下单");
            }
            // 判断是否已经出售
            if ($tradeStatus == TradeModel::T_TRADE_STATUS_DONE) {
                throw new Exception("商品已出售");
            }
            if (in_array($tradeStatus, TradeModel::T_TRADE_STATUS_COLLECT)) {
                throw new Exception("商品已下架");
            }

            // 判断出售是否结束
            $outtime = $mInfo['outtime'] ?? 0;
            if ($outtime < time()) {
                throw new Exception("交易已结束");
            }

            if ($mInfo['specuser'] != '' && $mInfo['specuser'] != $username) {
                throw new Exception("玩家已指定账户,不能购买!");
            }

            if ($tradeInfo['username'] == $username) {
                throw new Exception("不能购买自己发布的商品");
            }
            $config   = config('app.charge')[$model] ?? 0;
            $minPrice = $config['min_price'] ?? 0;
            $price    = $mInfo['price'] ?? 0;
            $charge   = (new TradeOrder())->getCharge($type, $price, (bool)$mInfo['specuser']);
            if ($price < 1 || $charge < 1 || $price < $minPrice) {
                throw new Exception("商品价格有误，请确认后再购买");
            }
            $game     = $tradeInfo['game'] ?? '';
            $serverid = $tradeInfo['serverid'] ?? 0;

            // 是否可以购买判断
            // 状态码和消息映射数组
            $puyStatus  = config($game . ".order_error_message");
            $res = GameTradeService::isbuyrole($game, $serverid, $username, $tradeInfo['sync_tradeno']);
            if (!isset($res['isbuy']) || $res['isbuy'] != 1) {
                $isbuy = $res['isbuy'] ?? -1;
                $msg   = $puyStatus[$isbuy] ?? '当前账号不能购买';
                throw new Exception($msg);
            }

            //购买角色判断
            if ($tradeInfo['type'] != TradeModel::TYPE_ROLES) {
                $roleLists = GameTradeService::getGameRoleList($game, $serverid, $username);
                if (!$roleLists) {
                    throw new Exception("请先在游戏内创建角色");
                }
            }

            $prefix  = strtoupper(substr($model, 5, 1));
            $orderno = generateOrderNo($prefix);
            if (TradeOrder::where(['orderno' => $orderno])->count() > 0) {
                throw new Exception("服务器繁忙，请稍后再试");
            }

            $orderdetails = [];
            switch ($type) {
                case TradeModel::TYPE_GOLD:
                    $orderdetails = [
                        'num'          => bcdiv($mInfo['num'] ?? 0, config($game . '.gold_num_ratio')), // 实际num为除以系数
                        'outtime'      => $outtime,
                        'sync_tradeno' => $tradeInfo['sync_tradeno'] ?? "",
                        'username'     => $tradeInfo['username'], // 卖家账号
                    ];
                    break;
                case TradeModel::TYPE_ROLES:
                    $orderdetails = [
                        'chaid'        => $mInfo['chaid'] ?? 0,
                        'chaname'      => $mInfo['chaname'] ?? "",
                        'job'          => $mInfo['job'] ?? 0,
                        'outtime'      => $outtime,
                        'sync_tradeno' => $tradeInfo['sync_tradeno'] ?? "",
                        'username'     => $tradeInfo['username'], // 卖家账号
                    ];
                    break;
                case TradeModel::TYPE_PROP:
                    $item = (new TradeItems())->getDetailsByItemid($game, $tradeInfo['serverid'], $mInfo['itemid'], ['itemid', 'itemname']);
                    if (!$item) {
                        throw new Exception("道具不存在，无法购买");
                    }
                    $orderdetails = [
                        'itemid'       => $item['itemid'] ?? 0,
                        'num'          => $mInfo['num'] ?? 0,
                        'goods_name'   => $item['itemname'] ?? '',
                        'outtime'      => $outtime,
                        'sync_tradeno' => $tradeInfo['sync_tradeno'] ?? "",
                        'username'     => $tradeInfo['username'], // 卖家账号
                    ];
                    break;
            }

            // 开始事务
            Db::startTrans();
            try {
                $data['game']          = $tradeInfo['game'];
                $data['serverid']      = $tradeInfo['serverid'];
                $data['tid']           = $tradeInfo['id'];
                $data['orderno']       = $orderno;
                $data['relationid']    = $mInfo['id'];
                $data['ordertype']     = $tradeInfo['type'];
                $data['charge']        = $charge;
                $data['price']         = $price;
                $data['amount']        = bcsub($price, $charge, 2);
                $data['tousername']    = $username;
                $data['addtime']       = time();
                $data['order_details'] = json_encode($orderdetails, JSON_UNESCAPED_UNICODE);
                TradeOrder::create($data);
                $M = app('app\\common\\model\\' . $model);
                $M::where('id', $mInfo['id'])->update([
                    'trade_status' => TradeModel::T_TRADE_STATUS_PLACE,
                    'pay_status'   => TradeModel::TYPE_PAY_STATUS_TODO,
                    'orderno'      => $orderno,
                ]);

                Db::commit();
            } catch (\Exception $e) {
                // 回滚事务
                Db::rollback();
                throw new Exception("服务器繁忙，请稍后再试");
            }
        } catch (Exception $e) {
            return $this->fail($e->getMessage());
        } finally {
            if ($fp !== false) {
                $obj->deleteLockFile($fp, $lockKey);
            }
        }

        return $this->success("success", [
            'url' => "/my/order/" . $orderno . ".html?mode=1",
        ]);
    }

    /**
     *  提交订单
     * @param $orderno
     * @return string
     * @throws \think\db\exception\DataNotFoundException
     * @throws \think\db\exception\DbException
     * @throws \think\db\exception\ModelNotFoundException
     */
    public function order($orderno = 0)
    {
        $username   = $this->request->member['username'] ?? "";
        $order = TradeOrder::where(['orderno' => $orderno, 'tousername' => $username])->find();
        if (!$order) {
            abort(404, '订单不存在');
        }
        $order = $order->toArray();

        $model    = TradeModel::getTypeTradeModel($order['ordertype']);
        $modelObj = app('app\\common\\model\\' . $model);
        $data     = $modelObj->getDetails($order['relationid'], $this->game, $this->serverid);

        $orderOuttime = (new TradeOrder())->getOrderOuttime(['order_addtime' => $order['addtime'], 'order_pay_status' => $order['pay_status']]);
        $order['outtime'] = bcadd($order['addtime'], config("app.order_outtime"));
        $order['outtime_text'] = $orderOuttime['order_outtime_text'] ?? "";
        $order['outtime_type'] = $orderOuttime['order_outtime_type'] ?? 0; // 延长订单过期时间 0-以订单过期时间为准
        $order['pay_status_text'] = TradeOrder::TO_PAY_STATUS[$order['pay_status']] ?? "";

        // 角色无需选择角色
        $roleLists = [];
        if ($order['ordertype'] != TradeModel::TYPE_ROLES) {
            $roleLists = GameTradeService::getGameRoleList($this->game, $this->serverid, $username);
        }

        $this->assign([
            "details" => $data['details'] ?? [],
            "order"   => $order,
            "query"   => http_build_query($this->request->param()),
            'item'    => $data['item'] ?? [],
            'roles'   => array_column($roleLists, 'chaname', 'chaid'),
        ]);
        return $this->fetch('my/order');
    }

    public function dopay()
    {
        $orderno = $this->request->get('orderno', '0');
        $paytype = $this->request->get('paytype', 'alipay');
        $order   = TradeOrder::where(['orderno' => $orderno])->find();
        if (!$order) {
            abort(404, '订单不存在');
        }
        $order = $order->toArray();
        // 修改为根据状态来
        if ($order['pay_status'] == TradeOrder::TO_PAY_STATUS_CACEL) {
            abort(404, '订单不存在');
        }

        if ($order['pay_status'] == TradeOrder::TO_PAY_STATUS_DONE) {
            abort(404, '订单不存在');
        }
        // 计算订单关闭时间，因接口传参是最小单位为分钟
        $outtime = bcadd($order['addtime'], config("app.order_outtime"));

        require_once root_path() . 'extend/alipay/pagepay/service/AlipayTradeService.php';
        require_once root_path() . 'extend/alipay/pagepay/buildermodel/AlipayTradePagePayContentBuilder.php';
        $config            = config("alipay");
        $out_trade_no      = trim($order['orderno']);
        $subject           = trim('创天互娱珍宝阁');
        $total_amount      = trim($order['price']);
        $body              = trim('创天互娱珍宝阁');
        $payRequestBuilder = new \AlipayTradePagePayContentBuilder();
        $payRequestBuilder->setBody($body);
        $payRequestBuilder->setSubject($subject);
        $payRequestBuilder->setTotalAmount($total_amount);
        $payRequestBuilder->setOutTradeNo($out_trade_no);
        $payRequestBuilder->setTimeExpire(date("Y-m-d H:i:s", $outtime));
        $aop      = new \AlipayTradeService($config);
        $response = $aop->pagePay($payRequestBuilder, $config['return_url'], $config['notify_url']);
        var_dump($response);
    }

    public function checkPay()
    {
        if (!$this->request->isPost()) {
            return $this->fail("非法请求");
        }
        $orderno = $this->request->post('orderno', 0);
        $order = (new TradeOrder())->getDetailsByWhere(['orderno' => $orderno]);
        if (!$order) {
            return $this->fail("订单不存在");
        }

        // 修改为根据状态来 提前5秒关闭支付 - 避免
        $maxtime = bcadd($order['addtime'], config('app.order_outtime'));
        if (bcsub($maxtime, time()) < 0) {
            return $this->fail("订单已关闭");
        }
        // 付款方式
        $paytype = $this->request->post('paytype', "");
        if ($paytype != "alipay") {
            return $this->fail("付款方式错误");
        }

        // 角色信息判断
        $chaid = (int)$this->request->post('chaid', 0);
        if ($order['ordertype'] != TradeModel::TYPE_ROLES) {
            if (!$chaid) {
                return $this->fail("请先选择角色");
            }

            $roleLists = GameTradeService::getGameRoleList($this->game, $this->serverid, $order['tousername']);
            $chaidLists = array_column($roleLists, 'chaname','chaid');
            if (!in_array($chaid, array_keys($chaidLists))) {
                return $this->fail("选择角色错误");
            }

            $chaname = $chaidLists[$chaid] ?? "";

            // 更新订单数据
            $res = TradeOrder::where('orderno', $orderno)->update(['tochaid' => $chaid, "tochaname" => $chaname]);
            if (false === $res) {
                return $this->fail("系统异常");
            }
        }

        $orderDetails = $order['order_details'] ?? [];
        $game       = $order['game'] ?? '';
        $puyStatus  =  config($game . ".order_error_message");
        $res = GameTradeService::isbuyrole($this->game, $this->serverid, $order['tousername'] ?? "", $orderDetails['sync_tradeno'] ?? "");
//            $res['isbuy'] = 1;
        if (!isset($res['isbuy']) || $res['isbuy'] != 1) {
            $isbuy = $res['isbuy'] ?? -1;
            $msg   = $puyStatus[$isbuy] ?? '当前账号不能购买';
            return $this->fail($msg);
        }
        return $this->success();
    }
}