<?php
namespace app\common\service;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;
use think\facade\Log;

class HttpRequestService
{
    protected $client;
    protected $config = [
        'timeout' => 5,
        'verify' => false,
    ];
    protected $game = "";

    public function __construct(string $game = "",array $config = [])
    {
        $this->client = new Client(array_merge($this->config, $config));
        $this->game = $game;
    }

    /**
     *  GET请求
     * @param string $url
     * @param array $options
     * @return false|mixed|string
     */
    public function get(string $url, array$options = [])
    {
        return $this->request('GET',$url, $options);
    }

    /**
     *  POST请求
     * @param string $url
     * @param array $options
     * @return false|mixed|string
     */
    public function post(string $url, array$options = [])
    {
        return $this->request('POST',$url, $options);
    }

    /**
     *  请求
     * @param string $method
     * @param string $url
     * @param array $options
     * @return false|mixed|string
     */
    private function request(string $method, string$url, array $options)
    {
        $start = microtime(true);
        $options = array_merge($this->config['headers'] ?? [], $options);

        try {
            $response =$this->client->request($method,$url, $options);
            $result =$response->getBody()->getContents();

            $result = iconv("GBK","utf-8//IGNORE", $result);
//            $result = mb_convert_encoding($result, 'UTF-8', 'GBK');
            $this->logRequest($method, $url, $options, $result, $start);
            return $result ? json_decode($result, true) : [];
        } catch (\Exception $e) {
            $this->logError($method, $url,$options, $e,$start);
            return false;
        }
    }

    /**
     *  成功日志
     * @param string $method
     * @param string $url
     * @param array $options
     * @param $response
     * @param float $start
     * @return void
     */
    private function logRequest(string $method, string $url, array $options, $response, float $start)
    {
        $duration = round((microtime(true) -$start) * 1000, 2);
        Log::channel('http')->info(
            sprintf(
                "执行时长：%s, 游戏标识：%s, 请求地址：%s, 请求方式：%s, 请求参数：%s, 响应信息：%s",
                $duration . 'ms',
                $this->game,
                $url,
                $method,
                json_encode($options, JSON_UNESCAPED_UNICODE),
                json_encode([
                    'body' => $response ? json_decode($response, true) : []
                ], JSON_UNESCAPED_UNICODE)
            )
        );
    }

    /**
     *  失败日志
     * @param string $method
     * @param string $url
     * @param array $options
     * @param GuzzleException $e
     * @param float $start
     * @return void
     */
    private function logError(string $method, string$url, array $options, \Exception $e, float $start)
    {
        $duration = round((microtime(true) -$start) * 1000, 2);
        Log::channel('http')->error(
            sprintf(
                "执行时长：%s, 游戏标识：%s, 请求地址：%s, 请求方式：%s, 请求参数：%s, 错误信息：%s",
                $duration . 'ms',
                $this->game,
                $url,
                $method,
                json_encode($options, JSON_UNESCAPED_UNICODE),
                json_encode([
                    'code' => $e->getCode(),
                    'message' => $e->getMessage(),
                    'line' => $e->getLine(),
                    'file' => $e->getFile(),
                    'trace' => $e->getTraceAsString()
                ], JSON_UNESCAPED_UNICODE)
            )
        );
    }
}
