<?php

namespace app\common\model;

use app\admin\model\Base;
use app\common\model\Trade as TradeModel;

class TradeRole extends Base
{
    protected $table = 'trade_role';

    // 是否为熊猫人：0-否 1-是
    const RTR_ISPANDA_NO = 0;
    const RTR_ISPANDA_YES = 1;
    const RTR_ISPANDA = [
        self::RTR_ISPANDA_NO => "否",
        self::RTR_ISPANDA_YES => "是"
    ];

    /**
     * 列表（分页）
     * @param  array   $where    条件
     * @param  array   $order    排序
     * @param  string  $fields   字段
     * @param  int     $limit    条数
     * @return array
     */
    public function listQuery($where=[], $order=['id'=>'desc'], $fields = '*', $limit=0)
    {
        $d     = request()->get();
        $limit = $limit > 0 ? $limit : (isset($d['limit']) ? intval($d['limit']) : 10);
        $rs    = $this->where($where)->order($order)->field($fields)->paginate($limit)->toArray();
        $data  = $rs['data'] ?? [];

        if ($data) {
            $tids       = array_column($data, 'tid');
            $orderLists = (new TradeOrder())->getOrderListsByTids($tids, ['tid', 'price', 'amount', 'charge', 'tousername']);
            $orderLists = array_column($orderLists, null, 'tid');
            // 获取同步状态
            $syncStatusLists = (new Trade())->whereIn('id', $tids)->column( 'sync_status', 'id');
            foreach ($data as &$v) {
                $res  = $this->handleRawData($v);
                $rows = $res['rows'] ?? [];
                unset($res);

                $order = $orderLists[$v['tid']] ?? [];
                if ($order) {
                    $v['charge'] = $order['charge'] ?? 0;
                    $v['amount'] = $order['amount'] ?? 0;
                } else {
                    $v['charge'] = $rows['charge'] ?? 0;
                    $v['amount'] = $rows['amount'] ?? 0;
                }
                $v['tousername']      = $order['tousername'] ?? '';
                $v['withdraw_status'] = $order['withdraw_status'] ?? 0;
                $v['outtime_text']    = $rows['outtime_text'] ?? '';
                $v['trade_status']    = $rows['trade_status'] ?? 0;
                $v['kill']            = $rows['kill'] ?? 0;
                $v['sync_status']     = $syncStatusLists[($rows['tid'] ?? 0)] ?? 0;
            }
        }

        $rs['data'] = $data;

        return $rs;
    }

    /**
     *  获取详情
     * @param $data
     * @param $fields
     * @return array
     */
    public function getDetailsByWhere($where = [], $fields = ['*'])
    {
        $details = self::field($fields)
            ->where($where)
            ->find();
        if (!$details) {
            return [];
        }

        return $details->toArray();
    }
    /**
     * 平台
     * @param array    $where
     * @param string[] $order
     * @param int      $limit
     * @param array    $extra
     *
     * @return array
     * @throws \think\db\exception\DbException
     */
    public function listIndexQuery($where =[], $order=['id'=>'desc'], $limit = 8, $extra = []): array
    {
        $d        = request()->get();
        $game     = $extra['game'] ?? "";
        $serverid = $extra['serverid'] ?? 0;

        $fields = [
            'id', 'game', 'tid', 'title', 'chalv', 'kill', 'price', 'grow', 'created_at', 'trade_status', 'job', 'family', 'pay_status'
        ];

        $rs   = self::field($fields)
            ->where($where)
            ->order($order)
            ->field($fields)
            ->paginate([
                'query'     => $d,
                'var_page'  => 'p',
                'list_rows' => $limit,
            ]);
        $page = $rs->render();
        $data = $rs->getCollection()->toArray();
        unset($rs);

        $lists = [];
        if ($data) {
            // 查询收藏数量
            $tids         = array_column($data, 'tid');
            $collectLists = (new TradeCollect())->getCollectLists($game, $serverid, $tids);

            // 获取同步状态
            $syncStatusLists = (new Trade())->whereIn('id', $tids)->column( 'sync_status', 'id');

            foreach ($data as $v) {
                $res = $this->handleRawData($v);
                $v   = $res['rows'] ?? [];
                unset($res);

                $collect = $collectLists[$v['tid']] ?? 0;
                $syncStatus = $syncStatusLists[$v['tid']] ?? 0;

                $lists[] = [
                    'id'           => $v['id'],
                    'tid'          => $v['tid'],
                    'name'         => $v['name'] ?? "",
                    'title'        => $v['title'],
                    'chalv'        => $v['chalv'] . "级",
                    'kill'         => $v['kill'],
                    'price'        => $v['price'],
                    'trade_status' => $v['trade_status'],
                    'outtime_type' => $v['outtime_type'] ?? 0,
                    'outtime_text' => $v['outtime_text'] ?? '',
                    'job_head'     => $v['job_head'] ?? '',
                    'collect'      => $collect,
                    'type'         => TradeModel::TYPE_ROLES,
                    'sync_status'  => $syncStatus,
                ];
            }
        }

        return [
            'lists' => $lists,
            'page'  => $page,
        ];
    }

    /**
     *  角色数据
     * @param string $game
     * @param int $serverid
     * @param array $tids
     * @return array
     */
    public function getRoleListsByTids(string $game = "", int $serverid = 0, array $tids = []): array
    {
        $lists = $this->field([
                "id", "game", "tid", "family", "job", "grow", "price", "chalv", "api_status", "pay_status",
                "trade_status", "outtime", "created_at", "pay_status", "orderno"
            ])
            ->where("game", $game)
            ->where("serverid", $serverid)
            ->whereIn("tid", $tids)
            ->select()
            ->toArray();
        if (!$lists) {
            return [];
        }

        $ordernos = array_values(array_filter(array_column($lists, 'orderno')));
        $orderLists = [];
        if ($ordernos) {
            $orderLists = (new TradeOrder())->getOrderListsByWhere([
                ['orderno', 'in', $ordernos],
            ], ['orderno', 'addtime', 'pay_status']);
            $orderLists = array_column($orderLists, null, 'orderno');
        }

        $data = [];
        foreach ($lists as $v) {
            $order = $orderLists[$v['orderno']] ?? [];
            $v['order_addtime'] = $order['addtime'] ?? 0;
            $v['order_pay_status'] = $order['pay_status'] ?? 0;

            $res = $this->handleRawData($v);
            $v = $res['rows'] ?? [];
            unset($res);
            $data[] = [
                'id'                 => $v['id'],
                'tid'                => $v['tid'],
                'name'               => $v['name'] ?? "",
                "price"              => $v['price'],
                "chalv"              => $v['chalv'],
                "created_at"         => $v['created_at'],
                "trade_status"       => $v['trade_status'],
                "trade_status_text"  => $v['trade_status_text'] ?? "",
                "outtime_type"       => $v['outtime_type'] ?? 0,
                "outtime_text"       => $v['outtime_text'] ?? "",
                "orderno"            => $v['orderno'] ?? "",
                "job_head"           => $v['job_head'],
                "order_outtime_type" => $v['order_outtime_type'] ?? 0,
                "order_outtime_text" => $v['order_outtime_text'] ?? "",
            ];
        }

        unset($lists);
        return $data;
    }

    /**
     *  数据处理
     * @param $data
     * @param $item
     * @return array
     */
    public function handleRawData($data = [], $item = []): array
    {
        $game        = $data['game'] ?? "";
        $jobLists    = config($game . ".job");
        $growLists   = config($game . ".grow");
        $familyLists = config($game . ".family");

        $charge = (new TradeOrder())->getCharge(TradeModel::TYPE_ROLES, ($data['price'] ?? 0), (bool)($data['specuser'] ?? ""));
        $jobDetails  = $jobLists[$data['family']] ?? [];
        $jobText     = $jobDetails[$data['job']]["name"] ?? "";
        $jobHead     = $jobDetails[$data['job']]["head"] ?? "";
        $growText    = $growLists[$data['grow']] ?? "";
        $familyText  = $familyLists[$data['family']] ?? "";
        $outtimeText = calculateRemainingTimeText(strtotime($data['created_at']), time());
        $tradeStatus = (new TradeModel())->getTradeStatus($data['trade_status'], $data['created_at'], $data['pay_status'] ?? 0);

        $orderOuttime               = (new TradeOrder())->getOrderOuttime($data);
        $data['order_outtime_type'] = $orderOuttime['order_outtime_type'] ?? 0;
        $data['order_outtime_text'] = $orderOuttime['order_outtime_text'] ?? "";
        $data['name']              = $jobText ? $jobText . "（" . ($growText) . "）" : "";
        $data['outtime_type']      = key($outtimeText);
        $data['outtime_text']      = current($outtimeText);
        $data['job_text']          = $jobText;
        $data['grow_text']         = $growText;
        $data['family_text']       = $familyText;
        $data['charge']            = $charge;
        $data['job_head']          = $jobHead;
        $data['kill']              = bcdiv($data['kill'] ?? 0, config($game . ".role_kill_ratio"), 2);
        $data['amount']            = bcsub($data['price'], $charge, 2);
        $data['trade_status']      = key($tradeStatus);
        $data['trade_status_text'] = current($tradeStatus);
        return ['rows' => $data, 'item' => $item];
    }
    /**
     *  角色详情信息
     * @param int $id
     * @param string $game
     * @param int $serverid
     *
     * @return array
     */
    public function getDetails(int $id, string $game = "", int $serverid = 0): array
    {
        // 获取详情
        $details = $this->getDetailsByWhere([
            'id' => $id, 'game' => $game, 'serverid' => $serverid,
        ], ["id", "game", "tid", "title", "job", "grow", "chaid", "chaname", "chalv", "kill", "family", "price", "created_at", "trade_status", "pay_status", "money"]);
        if ($details) {
            $res  = $this->handleRawData($details);
            $rows = $res['rows'] ?? [];
            unset($res);
            $details['job_head']     = $rows['job_head'] ?? "";
            $details['job_text']     = $rows['job_text'] ?? "";
            $details['grow_text']    = $rows['grow_text'] ?? "";
            $details['outtime_type'] = $rows['outtime_type'] ?? "";
            $details['outtime_text'] = $rows['outtime_text'] ?? "";
            $details['kill']         = $rows['kill'] ?? 0;
        }

        $stLists     = config($game . ".st");
        $tid         = $details['tid'] ?? 0;
        $roleDetails = $this->getRoleDetails($tid, $game, $serverid);

        // 获取数据同步状态
        $trade = (new Trade())->getDetailsByWhere(['id' => $tid, 'game' => $game, 'serverid' => $serverid], ['sync_status']);

        return [
            'details'           => $details,
            'trade'             => $trade,
            'warLists'          => $roleDetails['warLists'] ?? [],
            'equipLists'        => $roleDetails['equipLists'] ?? [],
            'stLists'           => $stLists,
            'packsackLists'     => $roleDetails['backpackLists'][0] ?? [], // 背包道具
            'roleBackpackLists' => $roleDetails['backpackLists'][1] ?? [], // 角色仓库
            'facadeLists'       => $roleDetails['facadeLists'] ?? [], // 外观
            'godPartLists'      => $roleDetails['godPartLists'] ?? [], // 天神属性列表
//            'godTotalLists'     => $roleDetails['godTotalLists'] ?? [], // 天神总属性
            'wingLists'         => $roleDetails['wingLists'] ?? [], // 翅膀属性列表
//            'wingTotalLists'    => $roleDetails['wingTotalLists'] ?? [], // 翅膀总属性
        ];
    }

    /**
     *  处理外观数据
     * @param array $equipFacadeLists
     * @param array $backpackFacadeLists
     * @return array
     */
    public function formatFacadData(array $equipFacadeLists = [], array $backpackFacadeLists = []): array
    {
        if (!$equipFacadeLists && !$backpackFacadeLists) {
            return [];
        }

        // 装备 - 外观
        ksort($equipFacadeLists);

        // 仓库 - 外观
        ksort($backpackFacadeLists);

        $facadeLists = $equipFacadeLists;
        if ($backpackFacadeLists) {
            foreach ($backpackFacadeLists as $key => $backpack) {
                $facadeLists[$key] = array_merge($facadeLists[$key] ?? [], $backpack);
            }
        }

        return $facadeLists;
    }

    /**
     *  获取角色背包详情
     * @param int $tid
     * @param string $game
     * @param int $serverid
     * @return array
     */
    public function getRoleDetails(int $tid = 0, string $game = "", int $serverid = 0): array
    {
        // 获取基础信息
        $warLists = (new TradeWar())->getTradeWarList($game, $serverid, $tid);
        // 获取穿戴装备
        $equipLists = (new TradeEquip())->getTradeEquipList($game, $serverid, $tid);
        // 获取道具
        $backpackLists = (new TradeBackPack())->getBackpackList($game, $serverid, $tid);
        // 获取天神数据
        $godLists = (new TradeGod())->getGodLists($game, $serverid, $tid);
        // 获取翅膀数据
        $wingLists = (new TradeWing())->getWingLists($game, $serverid, $tid);

        // 外观数据
        $facadeLists = $this->formatFacadData($equipLists['facade'] ?? [], $backpackLists['facade'] ?? []);
        unset($backpackLists['facade'], $equipLists['facade']);

        return [
            'warLists'       => $warLists,
            'equipLists'     => $equipLists,
            'backpackLists'  => $backpackLists,
            'facadeLists'    => $facadeLists,
            'godPartLists'   => $godLists['part'] ?? [],
//            'godTotalLists'  => $godLists['total'] ?? [],
            'wingLists'      => $wingLists['lists'] ?? [],
//            'wingTotalLists' => $wingLists['total'] ?? [],
        ];
    }

    /**
     *  获取列表
     * @param array $where
     * @param array $fields
     * @return array
     */
    public function getListsByWhere(array $where = [], array $fields = ['*']): array
    {
        return self::field($fields)
            ->where($where)
            ->select()
            ->toArray();
    }
}