<?php

namespace app\common\model;

use app\admin\model\Base;
use app\common\model\Trade as TradeModel;

class TradeProp extends Base
{
    protected $table = 'trade_prop';

    // 是否为熊猫人：0-否 1-是
    const PTR_ISPANDA_NO = 0;
    const PTR_ISPANDA_YES = 1;
    const PTR_ISPANDA = [
        self::PTR_ISPANDA_NO => "否",
        self::PTR_ISPANDA_YES => "是"
    ];

    // 可用职业：0-部分职业 1-全部职业
    const JOB_TYPE_PART = 0;
    const JOB_TYPE_ALL = 1;
    const JOB_TYPE = [
        self::JOB_TYPE_PART => "部分职业",
        self::JOB_TYPE_ALL  => "全部职业",
    ];

    /**
     * 列表（分页）
     * @param  array   $where    条件
     * @param  array   $order    排序
     * @param  string  $fields   字段
     * @param  int     $limit    条数
     * @return array
     */
    public function listQuery($where=[], $order=['id'=>'desc'], $fields = '*', $limit=0)
    {
        $d     = request()->get();
        $limit = $limit > 0 ? $limit : (isset($d['limit']) ? intval($d['limit']) : 10);
        $rs    = $this->where($where)->order($order)->field($fields)->paginate($limit)->toArray();
        $data  = $rs['data'] ?? [];

        if ($data) {
            $tids       = array_column($data, 'tid');
            $orderLists = (new TradeOrder())->getOrderListsByTids($tids, ['tid', 'price', 'amount', 'charge', 'tousername']);
            $orderLists = array_column($orderLists, null, 'tid');

            foreach ($data as &$v) {
                $res  = $this->handleRawData($v);
                $rows = $res['rows'] ?? [];
                unset($res);

                $order = $orderLists[$v['tid']] ?? [];
                if ($order) {
                    $v['charge'] = $order['charge'] ?? 0;
                    $v['amount'] = $order['amount'] ?? 0;
                } else {
                    $v['charge'] = $rows['charge'] ?? 0;
                    $v['amount'] = $rows['amount'] ?? 0;
                }

                $v['tousername']      = $order['tousername'] ?? '';
                $v['withdraw_status'] = $order['withdraw_status'] ?? 0;
                $v['outtime_text']    = $rows['outtime_text'] ?? '';
            }
        }

        $rs['data'] = $data;
        return $rs;
    }
    /**
     * 平台
     * @param array    $where
     * @param string[] $order
     * @param int      $limit
     * @param array    $extra
     *
     * @return array
     * @throws \think\db\exception\DataNotFoundException
     * @throws \think\db\exception\DbException
     * @throws \think\db\exception\ModelNotFoundException
     */
    public function listIndexQuery($where = [], $order=['id'=>'desc'], $limit = 8, $extra = []): array
    {
        $d        = request()->get();
        $game     = $extra['game'] ?? "";
        $serverid = $extra['serverid'] ?? 0;

        $fields = ['id', 'game', 'tid', 'num', 'itemid', 'price', "unit_price", 'created_at', 'trade_status', 'job', 'pay_status'];
        $rs     = self::field($fields)
            ->where($where)
            ->order($order)
            ->field($fields)
            ->paginate([
                'query'     => $d,
                'var_page'  => 'p',
                'list_rows' => $limit,
            ]);
        $page   = $rs->render();
        $data   = $rs->getCollection()->toArray();
        unset($rs);

        $lists = [];
        if ($data) {
            $itemids   = array_column($data, 'itemid');
            $itemLists   = (new TradeItems())
                ->getItemsListsByItemids($game, $serverid, $itemids, ['itemid', 'icon', 'itemname', 'iconset', 'og']);
            $itemLists = array_column($itemLists, null, 'itemid');

            $tids      = array_column($data, 'tid');
            // 获取职业
            $cjLists = (new TradeEquip())->getEquipListsByTids($game, $serverid, $tids, ['cj', 'itemid']);
            $cjLists = array_column($cjLists, "cj", 'itemid');

            // 查询收藏数量
            $collectLists = (new TradeCollect())->getCollectLists($game, $serverid, $tids);

            foreach ($data as $v) {
                $collect = $collectLists[$v['tid']] ?? 0;
                $v['cj'] = $cjLists[$v['itemid'] ?? 0] ?? 0;

                $res     = $this->handleRawData($v, $itemLists[$v['itemid'] ?? 0] ?? []);
                $v       = $res['rows'] ?? [];
                $item    = $res['item'] ?? [];
                unset($res);

                $lists[] = [
                    'id'           => $v['id'],
                    'tid'          => $v['tid'],
                    'itemname'     => $item['itemname'] ?? "",
                    'og_text'      => $item['og_text'] ?? "",
                    'iconx'        => $item['iconx'] ?? 0,
                    'icony'        => $item['icony'] ?? 0,
                    'iconset'      => $item['iconset'] ?? "",
                    'price'        => $v['price'],
                    'num'          => $v['num'],
                    'job_type'     => $item['job_type'] ?? 0,
                    'job_type_text'=> $item['job_type_text'] ?? "",
                    'job_text'     => $item['job_text'] ?? "",
                    'unit_price'   => $v['unit_price'],
                    'trade_status' => $v['trade_status'],
                    'outtime_type' => $v['outtime_type'] ?? 0,
                    'outtime_text' => $v['outtime_text'] ?? "",
                    'collect'      => $collect,
                    'type'         => TradeModel::TYPE_PROP,
                ];
            }
        }

        return [
            'lists' => $lists,
            'page'  => $page,
        ];
    }
    /**
     *  获取详情
     * @param $data
     * @param $fields
     * @return array
     */
    public function getDetailsByWhere($where = [], $fields = ['*'])
    {
        $details = self::field($fields)
            ->where($where)
            ->find();
        if (!$details) {
            return [];
        }

        return $details->toArray();
    }
    /**
     *  道具数据
     * @param string $game
     * @param int $serverid
     * @param array $tids
     * @return array
     */
    public function getPropListsByTids(string $game = "", int $serverid = 0, array $tids = []): array
    {
        $lists = $this->field([
                "id", "game", "tid", "itemid", "num", "price", "api_status", "pay_status",
                "trade_status", "outtime", "created_at", 'pay_status', "orderno"
            ])
            ->where("game", $game)
            ->where("serverid", $serverid)
            ->whereIn("tid", $tids)
            ->select()
            ->toArray();
        if (!$lists) {
            return [];
        }

        // 道具ID
        $itemids   = array_values(array_unique(array_column($lists, 'itemid')));
        $itemLists = (new TradeItems())->getItemsListsByItemids($game, $serverid, $itemids, ['itemid', 'icon', 'itemname', 'iconset', 'og']);
        $itemLists = array_column($itemLists, null, 'itemid');

        $ordernos = array_values(array_filter(array_column($lists, 'orderno')));
        $orderLists = [];
        if ($ordernos) {
            $orderLists = (new TradeOrder())->getOrderListsByWhere([
                ['orderno', 'in', $ordernos],
            ], ['orderno', 'addtime', 'pay_status']);
            $orderLists = array_column($orderLists, null, 'orderno');
        }

        $data = [];
        foreach ($lists as $v) {
            $order = $orderLists[$v['orderno']] ?? [];
            $v['order_addtime'] = $order['addtime'] ?? 0;
            $v['order_pay_status'] = $order['pay_status'] ?? 0;
            $res  = $this->handleRawData($v, $itemLists[$v['itemid']] ?? []);
            $v    = $res['rows'] ?? [];
            $item = $res['item'] ?? [];

            $data[] = [
                'id'                 => $v['id'],
                'tid'                => $v['tid'],
                'og_text'            => $item['og_text'] ?? "",
                'name'               => $item['itemname'] ?? "",
                'iconx'              => $item['iconx'] ?? 0,
                'icony'              => $item['icony'] ?? 0,
                'iconset'            => $item['iconset'] ?? "",
                "price"              => $v['price'],
                "created_at"         => $v['created_at'],
                "trade_status"       => $v['trade_status'],
                "trade_status_text"  => $v['trade_status_text'] ?? "",
                "outtime_type"       => $v['outtime_type'] ?? 0,
                "outtime_text"       => $v['outtime_text'] ?? "",
                "orderno"            => $v['orderno'] ?? "",
                "num"                => $v['num'] ?? 0,
                "order_outtime_type" => $v['order_outtime_type'] ?? 0,
                "order_outtime_text" => $v['order_outtime_text'] ?? "",
            ];
        }

        unset($lists);
        return $data;
    }

    /**
     *  数据处理
     * @param array $data
     * @param array $item
     * @return array
     */
    public function handleRawData(array $data = [], array $item = []): array
    {
        $game     = $data["game"] ?? "";
        $ogLists  = config($game . ".og");
        $jobBitLists      = config($game . ".job_bit");
        $dexColor = config($game . ".dex_color"); // 名称色值

        $binary = intToBinary($data['cj'] ?? 0);
        $job = [];
        foreach ($jobBitLists as $k => $v) {
            if (($binary[$k] ?? 0)) {
                $job[] = $v;
            }
        }

        // 公示/出售时间
        $outtimeText = calculateRemainingTimeText(strtotime($data['created_at']), time());
        $ogText      = $ogLists[($item['og'] ?? "")] ?? "";
        $icon        = calcXAndY($item['icon'] ?? 0);
        $tradeStatus = (new TradeModel())->getTradeStatus($data['trade_status'], $data['created_at'], $data['pay_status'] ?? 0);
        $charge = (new TradeOrder())->getCharge(TradeModel::TYPE_PROP, ($data['price'] ?? 0), (bool)($data['specuser'] ?? ""));

        $orderOuttime               = (new TradeOrder())->getOrderOuttime($data);
        $data['order_outtime_type'] = $orderOuttime['order_outtime_type'] ?? 0;
        $data['order_outtime_text'] = $orderOuttime['order_outtime_text'] ?? "";
        $data['outtime_type']      = key($outtimeText);
        $data['outtime_text']      = current($outtimeText);
        $data['trade_status']      = key($tradeStatus);
        $data['trade_status_text'] = current($tradeStatus);

        // 可用职业类型
        $jobType = ($data['cj'] ?? 0) == 16383 ? self::JOB_TYPE_ALL : self::JOB_TYPE_PART;

        $item['og_text']           = $ogText;
        $item['itemname']          = $item['itemname'] ?? "";
        $item['job_type']          = $jobType;
        $item['job_type_text']     = self::JOB_TYPE[$jobType] ?? "";
        $item['job_text']          = join("、", $job);
        $item['iconx']             = $icon['x'] ?? 0;
        $item['icony']             = $icon['y'] ?? 0;
        $item['iconset']           = $item['iconset'] ?? "";
        $item['dex_rgb']           = $dexColor[$item['dex'] ?? 0] ?? "";
        $item['text']              = itemFieldHandle($game, "text", $item);
        $data['charge']            = $charge;
        $data['amount']            = bcsub($data['price'], $charge, 2);

        return ['rows' => $data, 'item' => $item];
    }

    /**
     * 获取道具数据
     * @param int $id
     * @param string $game
     * @param int    $serverid
     *
     * @return array
     */
    public function getDetails(int $id = 0, string $game = "", int $serverid = 0): array
    {
        // 获取详情
        $details = $this->getDetailsByWhere([
            'id' => $id, 'game' => $game, 'serverid' => $serverid
        ], ["id", "game", "tid", "serverid", "itemid", "num", "unit_price", "price", "created_at", "trade_status", "job", "pay_status"]);

        $itemid      = $details['itemid'] ?? 0;
        $itemDetails = (new TradeItems())->getDetailsByItemid($game, $serverid, $itemid);
        // 获取职业
        $cjLists = (new TradeEquip())->getEquipListsByTids($game, $serverid, [$details['tid'] ?? 0], ['cj', 'itemid']);
        $cjLists = array_column($cjLists, "cj", 'itemid');

        $details['cj'] = $cjLists[$itemid] ?? 0;
        $res = $this->handleRawData($details, $itemDetails);

        return [
            'details'    => $res['rows'] ?? [],
            'item'       => $res['item'] ?? [],
        ];
    }
    /**
     *  获取列表
     * @param array $where
     * @param array $fields
     * @return array
     */
    public function getListsByWhere(array $where = [], array $fields = ['*']): array
    {
        return self::field($fields)
            ->where($where)
            ->select()
            ->toArray();
    }
}