<?php

namespace app\common\model;

use app\admin\model\Base;
use think\Exception;
use Throwable;

class TradeOrder extends Base
{
    protected $table = 'trade_order';
    // 转移状态：0-无 1-成功 2-失败
    const API_STATUS_NO = 0;
    const API_STATUS_SUCCESS = 1;
    const API_STATUS_FAIL = 2;
    const API_STATUS = [
        self::API_STATUS_NO => '— —',
        self::API_STATUS_SUCCESS => '转移成功',
        self::API_STATUS_FAIL => '转移失败'
    ];
    // 订单类型：1-角色 2-金币
    const ORDERTYPE_ROLES = 1;
    const ORDERTYPE_GOLD = 0;
    const ORDERTYPE_PROP = 2;
    const ORDERTYPE = [
        self::ORDERTYPE_ROLES => "角色订单",
        self::ORDERTYPE_GOLD  => "金币订单"
    ];

    // 支付状态：0-等待支付 1-支付成功 2-支付取消
    const TO_PAY_STATUS_TODO = 0;
    const TO_PAY_STATUS_DONE = 1;
    const TO_PAY_STATUS_CACEL = 2;
    const TO_PAY_STATUS = [
        self::TO_PAY_STATUS_TODO  => "待付款",
        self::TO_PAY_STATUS_DONE  => "已完成",
        self::TO_PAY_STATUS_CACEL => "已取消"
    ];
    
    // 提现状态：0-无 1-申请提现 2-提现成功 3-提现失败
    const TO_WITHDRAW_STATUS_CONNOT = -1;
    const TO_WITHDRAW_STATUS_NO = 0;
    const TO_WITHDRAW_STATUS_APPLY = 1;
    const TO_WITHDRAW_STATUS_SUCCESS = 2;
    const TO_WITHDRAW_STATUS_FAIL = 3;
    const TO_WITHDRAW_STATUS = [
        self::TO_WITHDRAW_STATUS_NO => '— —',
        self::TO_WITHDRAW_STATUS_APPLY => '申请提现',
        self::TO_WITHDRAW_STATUS_SUCCESS => '提现成功',
        self::TO_WITHDRAW_STATUS_FAIL => '提现失败',
    ];

    const WITHDRAW_STATUS = [
        self::TO_WITHDRAW_STATUS_CONNOT => "不可提现",
        self::TO_WITHDRAW_STATUS_NO => "待提现",
        self::TO_WITHDRAW_STATUS_APPLY => "已申请",
        self::TO_WITHDRAW_STATUS_SUCCESS => "已提现",
        self::TO_WITHDRAW_STATUS_FAIL => "提现失败",
    ];

    /**
     *  我的订单
     * @param $data
     * @param $order
     * @param $limit
     * @return array
     */
    public function listOrderQuery($data = [], $order=['id'=>'desc'], $limit = 8)
    {
        $d        = request()->get();
        $game     = $data['game'] ?? "";
        $serverid = $data['serverid'] ?? 0;
        $username = $data['username'] ?? 0;

        $key   = $d['key'] ?? '';
        $value = $d['value'] ?? '';
        if (in_array($key, ['price', 'outtime']) && $key && $value) {
            $order = [$key => $value];
        }

        $where = [
            'game'       => $game,
            'serverid'   => $serverid,
            'tousername' => $username,
            ['addtime', '>=', (time() - config("app.my_order_lists_time"))]
        ];


        $fields = ['id', 'orderno', 'ordertype', 'tid', 'price', 'relationid', 'pay_status', 'addtime', 'tousername'];
        $rs = self::field($fields)
            ->where($where)
            ->order($order)
            ->field($fields)
            ->paginate([
                'query' => $d,
                'var_page' => 'p',
                'list_rows' => $limit,
            ]);
        $page = $rs->render();
        $data = $rs->getCollection()->toArray();
        $lists = [];
        if ($data) {
            $tids = array_column($data, 'ordertype','tid');
            // 按照类型区分id
            $tidLists = [];
            foreach ($tids as $k => $v) {
                $tidLists[$v][] = $k;
            }

            // 金币信息
            $goldLists = (new TradeGold())->getGoldListsByTids($game, $serverid, $tidLists[Trade::TYPE_GOLD] ?? []);
            $goldLists = array_column($goldLists, null, 'tid');

            // 角色信息
            $roleLists = (new TradeRole())->getRoleListsByTids($game, $serverid, $tidLists[Trade::TYPE_ROLES] ?? []);
            $roleLists = array_column($roleLists, null, 'tid');

            // 道具信息
            $propLists = (new TradeProp())->getPropListsByTids($game, $serverid, $tidLists[Trade::TYPE_PROP] ?? []);
            $propLists = array_column($propLists, null, 'tid');

            // 数据处理
            foreach ($data as $v) {
                switch ($v['ordertype']) {
                    case Trade::TYPE_GOLD:
                        $item = $goldLists[$v['tid']] ?? [];
                        break;
                    case Trade::TYPE_ROLES:
                        $item = $roleLists[$v['tid']] ?? [];
                        break;
                    case Trade::TYPE_PROP:
                        $item = $propLists[$v['tid']] ?? [];
                        break;
                }

                $lists[] = [
                    'orderno' => $v['orderno'],
                    'ordertype' => $v['ordertype'],
                    'price' => $v['price'],
                    'addtime' => date("Y-m-d H:i:s", $v['addtime']),
                    'outtime' => date("Y-m-d H:i:s", ($v['addtime'] + config('app.order_outtime'))),
                    'name' => $item['name'] ?? "",
                    'og_text' => $item['og_text'] ?? "",
                    'iconx' => $item['iconx'] ?? 0,
                    'icony' => $item['icony'] ?? 0,
                    'iconset' => $item['iconset'] ?? "",
                    'job_head' => $item['job_head'] ?? "",
                    'relationid' => $v['relationid'] ?? 0,
                    'pay_status' => $v['pay_status'],
                    'pay_status_text' => TradeOrder::TO_PAY_STATUS[$v['pay_status']] ?? "",
                    'tousername' => $v['tousername'] ?? "",
                ];
            }
        }

        return [
            'lists' => $lists,
            'page' => $page,
        ];
    }


    /**
     *  获取有效订单数据
     * @param $tid
     * @param $ordertype
     * @return array
     * @throws \think\db\exception\DataNotFoundException
     * @throws \think\db\exception\DbException
     * @throws \think\db\exception\ModelNotFoundException
     */
    public function getOrderDetailsByTid($tid = 0, $ordertype = 0)
    {
        $details = $this->where('tid', $tid)
            ->where('ordertype', $ordertype)
            ->whereIn('pay_status', [self::TO_PAY_STATUS_TODO, self::TO_PAY_STATUS_DONE])
            ->find();
        if (!$details) {
            return [];
        }

        return $details->toArray();
    }

    /**
     *  获取有效订单数据
     * @param array $tids
     * @param array $fields
     * @return array
     */
    public function getOrderListsByTids(array $tids = [], array $fields = ['*']): array
    {
        return $this->field($fields)
            ->whereIn('tid', $tids)
            ->whereIn('pay_status', [self::TO_PAY_STATUS_TODO, self::TO_PAY_STATUS_DONE])
            ->select()->toArray();
    }


    /**
     *  获取订单，订单状态为：0-待支付 和 支付成功的订单
     * @param $type
     * @param $id
     * @return array
     */
    public function getOrderInfo($type, $id)
    {
        $data = $this->field(["id", "orderno", "charge", "price", "amount", "withdraw_status", "pay_status", 'addtime'])
            ->where("relationid", $id)
            ->where("ordertype", $type)
            ->whereIn("pay_status", [self::TO_PAY_STATUS_TODO, self::TO_PAY_STATUS_DONE])
            ->find();
        if (!$data) {
            return [];
        }
        return $data->toArray();
    }

    /**
     * 数据处理
     * @param array $data
     * @param array $details
     *
     * @return array|mixed
     */
    public function handleRawData($data = [], $details = [])
    {
        $data['ws_status'] = $this->getWithdrawStatus($data, $details);
        $data['ws_status_text'] = TradeOrder::WITHDRAW_STATUS[$data['ws_status']] ?? "";
        return $data;
    }

    /**
     *  获取状态值
     * @param array $data
     * @param array $details
     *
     * @return int
     */
    public function getWithdrawStatus($data = [], $details = []): int
    {
        if (!$data) {
            return self::TO_WITHDRAW_STATUS_CONNOT;
        }

        if ($details["trade_status"] != Trade::T_TRADE_STATUS_DONE) {
            return self::TO_WITHDRAW_STATUS_CONNOT;
        }

        if ($data['pay_status'] == TradeOrder::TO_PAY_STATUS_DONE && $data['withdraw_status'] == TradeOrder::TO_WITHDRAW_STATUS_NO) {
            return self::TO_WITHDRAW_STATUS_NO;
        }

        return $data['withdraw_status'];
    }

    /**
     * 取消订单
     * @param $data
     * @return void
     * @throws \Exception
     */
    public function cancelOrder($data = [])
    {
        $type         = $data['ordertype'] ?? 0;
        $id           = $data['relationid'] ?? 0;
        $orderId      = $data['id'] ?? 0;
        $orderDetails = $data['order_details'] ?? [];
        $outtime      = $orderDetails['outtime'] ?? 0;
        $syncTradeNo  = $orderDetails['sync_tradeno'] ?? "";

        $model    = Trade::getTypeTradeModel($type);
        $modelObj = app('app\\common\\model\\' . $model);

        // 取消订单
        TradeOrder::where("id", $orderId)->update(['pay_status' => TradeOrder::TO_PAY_STATUS_CACEL, 'cancel_time' => time()]);

        if (time() >= $outtime) {
            (new Trade())->closeTrade([
                'game'         => $data['game'] ?? "",
                'serverid'     => $data['serverid'] ?? 0,
                'tid'          => $data['tid'] ?? 0,
                'username'     => $data['tousername'] ?? "",
                'orderno'      => $data['orderno'] ?? "",
                'id'           => $id,
                'type'         => $type,
                'sync_tradeno' => $syncTradeNo,
                'trade_status' => Trade::T_TRADE_STATUS_SYSTEM_DOWN,
                'notice_type'  => TradeNotice::NOTICE_TYPE_ORDER_DOWN,
                'handle_type'  => Trade::OP_CANCEL_ORDER
            ]);
        } else {
            $modelObj::where('id', $id)
                ->where('trade_status', Trade::T_TRADE_STATUS_PLACE)
                ->update([
                    'trade_status' => Trade::T_TRADE_STATUS_SELL,
                    'pay_status'   => Trade::TYPE_PAY_STATUS_NO,
                    'orderno'      => ''
                ]);
        }
    }

    /**
     *  获取详情
     * @param $data
     * @param $fields
     * @return array
     */
    public function getDetailsByWhere($where = [], $fields = ['*'])
    {
        $details = self::field($fields)
            ->where($where)
            ->find();
        if (!$details) {
            return [];
        }

        $details = $details->toArray();
        $details['order_details'] = json_decode(($details['order_details'] ?? "") ?: '', true);
        return $details;
    }

    /**
     *  获取有效订单数据
     * @param array $where
     * @param array $fields
     * @return array
     */
    public function getOrderListsByWhere(array $where = [], array $fields = ['*']): array
    {
        return $this->field($fields)
            ->where($where)
            ->select()->toArray();
    }

    /**
     *  获取订单过期时间
     * @param array $data
     * @return array
     */
    public function getOrderOuttime(array $data = []): array
    {
        $orderAddtime     = $data['order_addtime'] ?? 0;
        $orderOuttimeType = 0;
        $orderOuttimeText = "";
        $payStatus        = $data['order_pay_status'] ?? 0;

        if (!$orderAddtime) {
            return [
                'order_outtime_type' => $orderOuttimeType,
                'order_outtime_text' => $orderOuttimeText,
            ];
        }

        $calcu            = calculateRemainingTime($orderAddtime, time(), config("app.order_outtime"));
        $orderOuttimeType = (key($calcu) || in_array($payStatus, [TradeOrder::TO_PAY_STATUS_DONE, TradeOrder::TO_PAY_STATUS_CACEL])) ? 0 : 1;
        $orderOuttimeText = current($calcu);
        return [
            'order_outtime_type' => $orderOuttimeType,
            'order_outtime_text' => $orderOuttimeText,
        ];
    }

    /**
     * 计算手续费
     * @param int $type
     * @param $price
     * @param $isAppoint
     * @return int|mixed|string
     * @throws \Exception
     */
    public function getCharge(int $type = 0, $price = 0, $isAppoint = false)
    {
        try {
            $model  = Trade::getTypeTradeModel($type);
            $config = config('app.charge')[$model] ?? 0;

            // 手续费比例
            $chargeRatio = $config['charge_ratio'] ?? 0;
            // 指定手续费比例
            $appointRatio = $config['appoint_ratio'] ?? 0;
            // 最低手续费
            $minCharge = $config['min_charge'] ?? 0;
            // 最高手续费
            $maxCharge = $config['max_charge'] ?? 0;

            if ($isAppoint) {
                $charge = bcmul($appointRatio, $price, 2);
            } else {
                $charge = bcmul($chargeRatio, $price, 2);
            }

            // 如果设置了最低手续费，手续费低于最低手续给，以最低手续费为准
            if ($minCharge > 0 && $charge < $minCharge) {
                $charge = $minCharge;
            }
            // 如果设置了最高手续费，手续费高于最高手续费，以最高手续费为准
            if ($maxCharge > 0 && $charge > $maxCharge) {
                $charge = $maxCharge;
            }
        } catch (Throwable $e) {
            return 0;
        }
        return $charge;
    }
}