<?php

namespace app\common\model;

use app\admin\model\Base;

class TradeNotice extends Base
{
    protected $table = 'trade_notice';

    const IS_READ_NO = 0;
    const IS_READ_YES = 1;

    // 商品退回
    const NOTICE_TYPE_FAILURE = 0;
    // 出售成功
    const NOTICE_TYPE_SUCCESS = 1;
    // 出售结束
    const NOTICE_TYPE_ORDER_DOWN = 2;
    // 商品下架
    const NOTICE_TYPE_OWNER_DOWN = 3;

    // 站内信
    const NOTICE_TYPE = [
        self::NOTICE_TYPE_FAILURE    => [
            'title'   => "商品退回",
            /**
             * %s - 类型 - 商品名，金币类型：金币+数量，角色类型：角色+角色名称，道具类型：道具-道具名称
             * %s - 商品详情地址
             * %s - 客服地址
             */
            'content' => "%s未通过公示期%s，商品已退回游戏，请前往游戏内角色列表/角色邮箱查收。如有疑问可联系官方客服咨询%s"
        ],
        self::NOTICE_TYPE_SUCCESS    => [
            'title'   => "出售成功",
            /**
             * %s - 类型 - 商品名，金币类型：金币+数量，角色类型：角色+角色名称，道具类型：道具-道具名称
             * %s - 商品详情地址
             * %s - 客服地址
             */
            'content' => "%s出售成功%s！如有疑问可联系官方客服咨询%s"
        ],
        self::NOTICE_TYPE_ORDER_DOWN => [
            'title'   => "出售结束",
            /**
             * %s - 类型 - 商品名，金币类型：金币+数量，角色类型：角色+角色名称，道具类型：道具-道具名称
             * %s - 商品详情地址
             * %s - 客服地址
             */
            'content' => "%s出售期已结束%s，商品已退回游戏，请前往游戏内角色列表/角色邮箱查收。如有疑问可联系官方客服咨询%s"
        ],
        self::NOTICE_TYPE_OWNER_DOWN => [
            'title'   => "商品下架",
            /**
             * %s - 类型 - 商品名，金币类型：金币+数量，角色类型：角色+角色名称，道具类型：道具-道具名称
             * %s - 商品详情地址
             * %s - 客服地址
             */
            'content' => "%s已完成下架%s，商品已退回游戏，请前往游戏内角色列表/角色邮箱查收。如有疑问可联系官方客服咨询%s"
        ],
    ];

    /**
     * 列表（分页）
     * @param  array   $where    条件
     * @param  array   $order    排序
     * @param  string  $fields   字段
     * @param  int     $limit    条数
     * @return array
     */
    public function listQuery($where=[], $order=['id'=>'desc'], $fields = '*', $limit=0)
    {
        $d = request()->get();
        $limit  = $limit>0 ? $limit : (isset($d['limit']) ? intval($d['limit']) : 10);
        $rs = $this->where($where)->order($order)->field($fields)->paginate($limit)->toArray();
        $data = $rs['data'] ?? [];

        $rs['data'] = $data;
        return $rs;
    }
    /**
     *  站内信
     * @param array $data
     * @param $order
     * @param $limit
     * @return array
     * @throws \think\db\exception\DbException
     */
    public function listOrderQuery(array $data, $order=['id'=>'desc'], $limit = 8)
    {
        $d        = request()->get();
        $game     = $data['game'] ?? "";
        $serverid = $data['serverid'] ?? 0;
        $username = $data['username'] ?? "";

        $where = [
            'game'     => $game,
            'serverid' => $serverid,
            'username' => $username,
        ];

        $fields = ['id', 'tid', 'content', 'created_at', 'is_read', 'trade_type', 'type', 'orderno'];
        $rs     = self::field($fields)
            ->where($where)
            ->whereNull('deleted_at')
            ->order($order)
            ->field($fields)
            ->paginate([
                'query'     => $d,
                'var_page'  => 'p',
                'list_rows' => $limit,
            ]);
        $page   = $rs->render();
        $lists  = $rs->getCollection()->toArray();

        $data = [];
        if ($lists) {

            $tids = array_column($lists, 'trade_type', 'tid');

            // 按照类型区分id
            $tidLists = [];
            foreach ($tids as $k => $v) {
                $tidLists[$v][] = $k;
            }

            $goldIds   = $tidLists[Trade::TYPE_GOLD] ?? [];
            $goldLists = [];
            if ($goldIds) {
                $goldLists = (new TradeGold())->getListsByWhere([['tid', "IN", $goldIds]], ["id", "tid", "num"]);
                $goldLists = array_column($goldLists, null, 'tid');
            }

            $roleIds   = $tidLists[Trade::TYPE_ROLES] ?? [];
            $roleLists = [];
            if ($roleIds) {
                $roleLists = (new TradeRole())->getListsByWhere([['tid', "IN", $roleIds]], ["id", "tid", "chaname"]);
                $roleLists = array_column($roleLists, null, "tid");
            }

            $propIds   = $tidLists[Trade::TYPE_PROP] ?? [];
            $propLists = [];
            if ($propIds) {
                $propLists = (new TradeProp())->getListsByWhere([['tid', "IN", $propIds]], ["id", "tid", "num", "itemid"]);
                $itemIds   = array_column($propLists, 'itemid');
                $propLists = array_column($propLists, null, 'tid');
                $itemLists = [];
                if ($itemIds) {
                    $itemLists = (new TradeItems())->getItemsListsByItemids($game, $serverid, $itemIds, ['itemid', 'itemname']);
                    $itemLists = array_column($itemLists, "itemname", 'itemid');
                }
            }

            $extend = [];
            foreach ($lists as $v) {
                switch ($v['trade_type']) {
                    case Trade::TYPE_GOLD:
                        $item                  = $goldLists[$v['tid']] ?? [];
                        $extend["relation_id"] = $item['id'] ?? 0;
                        $extend["num"]         = $item["num"] ?? 0;
                        break;
                    case Trade::TYPE_ROLES:
                        $item                  = $roleLists[$v['tid']] ?? [];
                        $extend["relation_id"] = $item['id'] ?? 0;
                        $extend["chaname"]     = $item["chaname"] ?? "";
                        break;
                    case Trade::TYPE_PROP:
                        $item                  = $propLists[$v['tid']] ?? [];
                        $itemid                = $item['itemid'] ?? 0;
                        $extend["relation_id"] = $item['id'] ?? 0;
                        $extend["itemname"]    = $itemLists[$itemid] ?? "";
                        break;
                }

                $content = $this->handleRawContent($v['content'], $v['trade_type'], $v['type'], $v['orderno'], $extend);
                $data[]  = [
                    'id'         => $v['id'],
                    'title'      => self::NOTICE_TYPE[$v['type']]['title'] ?? "",
                    'created_at' => $v['created_at'],
                    'is_read'    => $v['is_read'],
                    'content'    => $content,
                ];
            }
        }

        return [
            'lists' => $data,
            'page'  => $page,
        ];
    }

    /**
     *  获取详情
     * @param array $where
     * @param array $fields
     * @return array
     */
    public function getDetailsByWhere(array $where=[], array $fields = ['*']): array
    {
        $details = $this->field($fields)->where($where)->find();
        if (!$details) {
            return [];
        }
        return $details->toArray();
    }

    /**
     * 添加站内信
     * @param string $game
     * @param int $serverid
     * @param int $tid
     * @param string $username
     * @param string $orderno
     * @param int $type
     * @param int $noticeType
     * @param array $entend
     * @return int|string
     */
    public function addNotice(
        string $game = "", int $serverid = 0,
        int    $tid = 0, string $username = "", string $orderno = "",
        int    $type = 0, int $noticeType = 0
    )
    {
        $notice = self::NOTICE_TYPE[$noticeType] ?? [];
        return self::insertGetId([
            'game'       => $game,
            'serverid'   => $serverid,
            'tid'        => $tid,
            'username'   => $username,
            'orderno'    => $orderno,
            'trade_type' => $type,
            'type'       => $noticeType,
            'content'    => $notice['content'] ?? "",
        ]);
    }

    /**
     *  站内信正文处理
     * @param $content
     * @param $extend
     * @return string
     */
    public function handleRawContent(string $content = "", int $tradeType = 0, int $type = 0, string $orderno = "", array $extend = []): string
    {
        $tradeTypeText = Trade::TYPE[$tradeType] ?? "";
        $detailsUrl    = Trade::JUMP_DETAILS[$tradeType] ?? "";
        $relationId    = $extend['relation_id'] ?? "";
        $chaname       = $extend['chaname'] ?? "";
        $num           = $extend['num'] ?? "";
        $itemname      = $extend['itemname'] ?? "";

        $kfUrl         = config("app.kf_url");
        $kfLabel = "";
        if ($kfUrl) {
            $kfLabel       = sprintf("<a href='%s' target='_blank' style='color: #00a0e9;'>%s</a>",
                $kfUrl
            , "--->>>客服服务中心<<<---");
        }

        $detailsUrlLabel = "";
        if ($detailsUrl && $relationId) {
            $detailsUrl = sprintf($detailsUrl, $relationId);
            $detailsUrlLabel = sprintf("<a href='%s' target='_blank' style='color: #00a0e9;'>%s</a>",
                $detailsUrl,
                "-->>查看详情<<--"
            );
        }

        $goodsName = "";
        switch ($tradeType) {
            case Trade::TYPE_GOLD:
                $goodsName = $tradeTypeText . "-" . $num . "万";
                break;
            case Trade::TYPE_ROLES:
                $goodsName = $tradeTypeText . "-" . $chaname;
                break;
            case Trade::TYPE_PROP:
                $goodsName = $tradeTypeText . "-" . $itemname;
                break;
        }

        switch ($type) {
            case self::NOTICE_TYPE_SUCCESS:
            case self::NOTICE_TYPE_FAILURE:
            case self::NOTICE_TYPE_ORDER_DOWN:
            case self::NOTICE_TYPE_OWNER_DOWN:
                $content = sprintf($content, $goodsName, $detailsUrlLabel, $kfLabel);
                break;
        }

        return $content;
    }
}