<?php

namespace app\common\model;

use app\admin\model\Base;
use app\common\model\Trade as TradeModel;

class TradeGold extends Base
{
    protected $table = 'trade_gold';

    /**
     * 列表（分页）
     * @param  array   $where    条件
     * @param  array   $order    排序
     * @param  string  $fields   字段
     * @param  int     $limit    条数
     * @return array
     */
    public function listQuery($where=[], $order=['id'=>'desc'], $fields = ['*'], $limit=0)
    {
        $d = request()->get();
        $limit  = $limit>0 ? $limit : (isset($d['limit']) ? intval($d['limit']) : 10);
        $rs = $this->where($where)->order($order)->field($fields)->paginate($limit)->toArray();
        $data = $rs['data'] ?? [];

        if ($data) {
            $tids = array_column($data, 'tid');
            $orderLists = (new TradeOrder())->getOrderListsByTids($tids, ['tid', 'price', 'amount', 'charge', 'tousername']);
            $orderLists = array_column($orderLists, null,'tid');

            foreach ($data as &$v) {
                $res = $this->handleRawData($v);
                $rows = $res['rows'] ?? [];
                unset($res);

                $order = $orderLists[$v['tid']] ?? [];
                if ($order) {
                    $v['charge'] = $order['charge'] ?? 0;
                    $v['amount'] = $order['amount'] ?? 0;
                } else {
                    $v['charge'] = $rows['charge'] ?? 0;
                    $v['amount'] = $rows['amount'] ?? 0;
                }

                $v['tousername']    = $order['tousername'] ?? '';
                $v['withdraw_status']  = $order['withdraw_status'] ?? 0;
                $v['outtime_text']  = $rows['outtime_text'] ?? '';
                $v['trade_status']  = $rows['trade_status'] ?? 0;
            }
        }

        $rs['data'] = $data;
        return $rs;
    }

    /**
     *  金币列表获取
     * @param array    $where
     * @param string[] $order
     * @param int      $limit
     * @param array    $extra
     *
     * @return array
     * @throws \think\db\exception\DbException
     */
    public function listIndexQuery($where = [], $order=['id'=>'desc'], $limit = 8, $extra = []): array
    {
        $d = request()->get();
        $game = $extra['game'] ?? "";
        $serverid = $extra['serverid'] ?? 0;

        $fields = ['id', 'num', 'price', 'tid', "unit_price",'created_at', 'trade_status', 'pay_status'];
        $rs = self::field($fields)
            ->where($where)
            ->order($order)
            ->field($fields)
            ->paginate([
                'query' => $d,
                'var_page' => 'p',
                'list_rows' => $limit,
            ]);
        $page = $rs->render();
        $data = $rs->getCollection()->toArray();
        unset($rs);

        $lists = [];
        if ($data) {
            // 查询收藏数量
            $tids         = array_column($data, 'tid');
            $collectLists = (new TradeCollect())->getCollectLists($game, $serverid, $tids);

            foreach ($data as $v) {
                $res = $this->handleRawData($v);
                $v = $res['rows'] ?? [];
                unset($res);
                $collect    = $collectLists[$v['tid']] ?? 0;

                $lists[] = [
                    'id'           => $v['id'],
                    'tid'          => $v['tid'],
                    'name'         => $v['name'] ?? '',
                    'unit_price'   => $v['unit_price'] ?? "0.00",
                    'price'        => $v['price'],
                    'trade_status' => $v['trade_status'],
                    'outtime_type' => $v['outtime_type'] ?? 0,
                    'outtime_text' => $v['outtime_text'] ?? "",
                    'collect'      => $collect,
                    'type'         => TradeModel::TYPE_GOLD,
                ];
            }
        }

        return [
            'lists' => $lists,
            'page' => $page,
        ];
    }
    /**
     *  获取详情
     * @param $data
     * @param $fields
     * @return array
     */
    public function getDetailsByWhere($where = [], $fields = ['*'])
    {
        $details = self::field($fields)
            ->where($where)
            ->find();
        if (!$details) {
            return [];
        }

        return $details->toArray();
    }
    /**
     *  获取金币信息
     * @param string $game
     * @param int $serverid
     * @param array $tids
     * @return array
     */
    public function getGoldListsByTids(string $game = "", int $serverid = 0, array $tids = []): array
    {
        if (!$tids) {
            return [];
        }

        $lists = $this->field(["id", "tid", "num", "unit_price", "price", "api_status", "pay_status", "trade_status", "outtime", "created_at", "orderno"])
            ->where("game", $game)
            ->where("serverid", $serverid)
            ->whereIn("tid", $tids)
            ->select()
            ->toArray();
        if (!$lists) {
            return [];
        }

        $ordernos = array_values(array_filter(array_column($lists, 'orderno')));
        $orderLists = [];
        if ($ordernos) {
            $orderLists = (new TradeOrder())->getOrderListsByWhere([
                ['orderno', 'in', $ordernos],
            ], ['orderno', 'addtime', 'pay_status']);
            $orderLists = array_column($orderLists, null, 'orderno');
        }

        $data = [];
        foreach ($lists as $v) {
            $order = $orderLists[$v['orderno']] ?? [];
            $v['order_addtime'] = $order['addtime'] ?? 0;
            $v['order_pay_status'] = $order['pay_status'] ?? 0;
            $res = $this->handleRawData($v);
            $v = $res['rows'] ?? [];
            unset($res);
            $data[] = [
                "id"                 => $v['id'],
                "tid"                => $v['tid'],
                "name"               => $v['name'] ?? "",
                "price"              => $v['price'],
                "created_at"         => $v['created_at'],
                "trade_status"       => $v['trade_status'],
                "trade_status_text"  => $v['trade_status_text'] ?? "",
                "outtime_type"       => $v['outtime_type'] ?? 0,
                "outtime_text"       => $v['outtime_text'] ?? "",
                "orderno"            => $v['orderno'] ?? "",
                "num"                => $v['num'] ?? 0,
                "order_outtime_type" => $v['order_outtime_type'] ?? 0,
                "order_outtime_text" => $v['order_outtime_text'] ?? "",
            ];
        }

        unset($lists);
        return $data;
    }

    /**
     *  名称
     * @param $num
     * @return string
     */
    public function getGoldName($num = 0)
    {
        return "金币-" . $num . "万";
    }

    /**
     *  数据处理
     * @param array $data
     * @param array $item
     * @return array
     */
    public function handleRawData(array $data = [], array $item = []): array
    {
        $charge = (new TradeOrder())->getCharge(TradeModel::TYPE_GOLD, ($data['price'] ?? 0), (bool)($data['specuser'] ?? ""));
        $remainText  = calculateRemainingTimeText(strtotime($data['created_at']), time());
        $tradeStatus = (new Trade())->getTradeStatus(
            $data['trade_status'],
            $data['created_at'],
            $data['pay_status']
        );

        $orderOuttime               = (new TradeOrder())->getOrderOuttime($data);
        $data['order_outtime_type'] = $orderOuttime['order_outtime_type'] ?? 0;
        $data['order_outtime_text'] = $orderOuttime['order_outtime_text'] ?? "";
        $data['outtime_type']       = key($remainText);
        $data['outtime_text']       = current($remainText);
        $data['name']               = $this->getGoldName($data['num']);
        $data['charge']             = $charge;
        $data['amount']             = bcsub($data['price'], $charge, 2);
        $data['trade_status']       = key($tradeStatus);
        $data['trade_status_text']  = current($tradeStatus);

        return ['rows' => $data, 'item' => $item];
    }
    /**
     * 获取金币数据
     * @param int $id
     * @param string $game
     * @param int $serverid
     *
     * @return array[]
     */
    public function getDetails(int $id, string $game = "", int $serverid = 0): array
    {
        // 获取详情
        $details = $this->getDetailsByWhere([
            'id' => $id, 'game' => $game, 'serverid' => $serverid
        ], ["id", "num", "tid", "unit_price", "price", "created_at", "trade_status", "pay_status"]);

        $res = $this->handleRawData($details);
        $details = $res['rows'] ?? [];
        unset($res);
        return [
            'details' => $details
        ];
    }

    public function getCloseTradeLists()
    {
        return self::where('outtime', '<', time())
                ->where('outtime', '>', time() - 1800)
                ->where('trade_status', '!=', Trade::T_TRADE_STATUS_DOWN)
                ->update(['trade_status' => Trade::T_TRADE_STATUS_DOWN]);
    }

    /**
     *  获取列表
     * @param array $where
     * @param array $fields
     * @return array
     */
    public function getListsByWhere(array $where = [], array $fields = ['*']): array
    {
        return self::field($fields)
            ->where($where)
            ->select()
            ->toArray();
    }
}