<?php

namespace app\common\model;

use app\admin\model\Base;

class TradeCollect extends Base
{
    protected $table = 'trade_collect';

    /**
     *  判断是否收藏
     * @param string $game
     * @param int $serverid
     * @param string $username
     * @param int $id
     * @param int $type
     * @return bool
     */
    public function isCollect(string $game = "", int $serverid= 0, string $username = "", int $id = 0, int $type = 0): bool
    {
        if (!$username) {
            return false;
        }

        $data = self::where([
            'game'       => $game,
            'serverid'   => $serverid,
            'username'   => $username,
            'type'       => $type,
            'relationid' => $id
        ])->count();

        return (bool)$data;
    }

    /**
     *  获取收藏数量
     * @param string $game
     * @param int $serverid
     * @param array $tids
     * @return array
     */
    public function getCollectLists(string $game = "", int $serverid= 0, array $tids = [])
    {
        $data = self::field(["tid", "count(1) as num"])->where([
            'game' => $game,
            'serverid' => $serverid,
        ])->whereIn('tid', $tids)->group(['tid'])->select()->toArray();
        return array_column($data, 'num', 'tid');
    }

    /**
     *  获取收藏列表
     * @param $data
     * @param $order
     * @param $limit
     * @return array[]
     */
    public function listCollectQuery(array $data = [], $order=['id'=>'desc'], $limit = 8)
    {
        $d        = request()->get();
        $game     = $data['game'] ?? "";
        $serverid = $data['serverid'] ?? 0;
        $username      = $data['username'] ?? "";

        $key   = $d['key'] ?? '';
        $value = $d['value'] ?? '';
        if (in_array($key, ['price', 'outtime']) && $key && $value) {
            $order = [$key => $value];
        }

        $where = [
            'game'     => $game,
            'serverid' => $serverid,
            'username' => $username,
            ['created_at', '>=', date("Y-m-d H:i:s", (time() - config("app.my_collect_lists_time")))]
        ];


        $fields = ['id', 'type', 'tid', 'relationid'];
        $rs     = self::field($fields)
            ->where($where)
            ->order($order)
            ->field($fields)
            ->paginate([
                'query'     => $d,
                'var_page'  => 'p',
                'list_rows' => $limit,
            ]);
        $page   = $rs->render();
        $data   = $rs->getCollection()->toArray();
        $lists  = [];
        if ($data) {
            $tids = array_column($data, 'type', 'tid');
            // 按照类型区分id
            $tidLists = [];
            foreach ($tids as $k => $v) {
                $tidLists[$v][] = $k;
            }

            // 金币信息
            $goldLists = (new TradeGold())->getGoldListsByTids($game, $serverid, $tidLists[Trade::TYPE_GOLD] ?? []);
            $goldLists = array_column($goldLists, null, 'tid');

            // 角色信息
            $roleLists = (new TradeRole())->getRoleListsByTids($game, $serverid, $tidLists[Trade::TYPE_ROLES] ?? []);
            $roleLists = array_column($roleLists, null, 'tid');

            // 道具信息
            $propLists = (new TradeProp())->getPropListsByTids($game, $serverid, $tidLists[Trade::TYPE_PROP] ?? []);
            $propLists = array_column($propLists, null, 'tid');

            // 数据处理
            foreach ($data as $v) {
                switch ($v['type']) {
                    case Trade::TYPE_GOLD:
                        $item = $goldLists[$v['tid']] ?? [];
                        break;
                    case Trade::TYPE_ROLES:
                        $item = $roleLists[$v['tid']] ?? [];
                        break;
                    case Trade::TYPE_PROP:
                        $item = $propLists[$v['tid']] ?? [];
                        break;
                }

                if (!$item) {
                    continue;
                }

                $lists[] = [
                    'tid'               => $v['tid'],
                    'type'              => $v['type'],
                    'price'             => $item['price'],
                    'name'              => $item['name'] ?? "",
                    'chalv'             => $item['chalv'] ?? 0,
                    'og_text'           => $item['og_text'] ?? "",
                    'iconx'             => $item['iconx'] ?? 0,
                    'icony'             => $item['icony'] ?? 0,
                    'iconset'           => $item['iconset'] ?? "",
                    'job_head'          => $item['job_head'] ?? "",
                    'relationid'        => $v['relationid'] ?? 0,
                    'num'               => $item['num'] ?? 0,
                    'trade_status'      => $item['trade_status'] ?? 0,
                    'trade_status_text' => $item['trade_status_text'] ?? "",
                    'outtime_type'      => $item['outtime_type'] ?? 0,
                    'outtime_text'      => $item['outtime_text'] ?? "",
                    'order_outtime_type'=> $item['order_outtime_type'] ?? 0,
                    'order_outtime_text'=> $item['order_outtime_text'] ?? "",
                ];
            }
        }

        return [
            'lists' => $lists,
            'page'  => $page,
        ];
    }
}