<?php

namespace app\common\model;

use app\admin\model\Base;
use app\common\model\Trade as TradeModel;
use app\common\service\GameTradeService;
use think\Exception;
use think\facade\Db;

class Trade extends Base
{
    protected $table = 'trade';
    // 设置主键
    protected $pk = 'id';
    //访问器
    // 交易类型：0-金币 1-角色 2-道具
    const TYPE_GOLD = 0;
    const TYPE_ROLES = 1;
    const TYPE_PROP = 2;
    const TYPE = [
        self::TYPE_GOLD  => "金币",
        self::TYPE_ROLES => "角色",
        self::TYPE_PROP  => "道具",
    ];
    // 同步状态：0-等待同步 1-同步中 2-同步成功 3-同步失败
    const T_SYNC_STATUS_TODO = 0;
    const T_SYNC_STATUS_DOING = 1;
    const T_SYNC_STATUS_SUCCESS = 2;
    const T_SYNC_STATUS_FAIL = 3;
    const T_SYNC_STATUS = [
        self::T_SYNC_STATUS_TODO    => '等待同步',
        self::T_SYNC_STATUS_DOING   => '同步中',
        self::T_SYNC_STATUS_SUCCESS => '同步成功',
        self::T_SYNC_STATUS_FAIL    => '同步失败',
    ];

    // 交易状态：0-数据同步中(上架) 1-公示中 2-售卖中 3-被下单 4-已出售 5-下架(玩家) 6-下架(系统) 7-下架(后台) 8-已结束
    const T_TRADE_STATUS_UP = 0;
    const T_TRADE_STATUS_PUB = 1;
    const T_TRADE_STATUS_SELL = 2;
    const T_TRADE_STATUS_PLACE = 3;
    const T_TRADE_STATUS_DONE = 4;
    const T_TRADE_STATUS_DOWN = 5;
    const T_TRADE_STATUS_SYSTEM_DOWN = 6;
    const T_TRADE_STATUS_ADMIN_DOWN = 7;
    const T_TRADE_STATUS = [
        self::T_TRADE_STATUS_UP          => "数据同步中",
        self::T_TRADE_STATUS_PUB         => "公示中",
        self::T_TRADE_STATUS_SELL        => "售卖中",
        self::T_TRADE_STATUS_PLACE       => "被下单",
        self::T_TRADE_STATUS_DONE        => "已出售",
        self::T_TRADE_STATUS_DOWN        => "已下架",
        self::T_TRADE_STATUS_SYSTEM_DOWN => "已下架(系统)",
        self::T_TRADE_STATUS_ADMIN_DOWN  => "已下架(后台)",
    ];
    // 下架
    const T_TRADE_STATUS_COLLECT = [
        self::T_TRADE_STATUS_DOWN,
        self::T_TRADE_STATUS_SYSTEM_DOWN,
        self::T_TRADE_STATUS_ADMIN_DOWN,
    ];

    // 支付状态：0-无 1-待支付 2-已支付
    const TYPE_PAY_STATUS_NO = 0;
    const TYPE_PAY_STATUS_TODO = 1;
    const TYPE_PAY_STATUS_DONE = 2;

    // 支付状态
    const TYPE_PAY_STATUS = [
        self::TYPE_PAY_STATUS_NO   => "— —",
        self::TYPE_PAY_STATUS_TODO => "待支付",
        self::TYPE_PAY_STATUS_DONE => "已支付"
    ];

    // 转移状态：0-无 1-成功 2-失败
    const T_API_STATUS_NO = 0;
    const T_API_STATUS_SUCCESS = 1;
    const T_API_STATUS_FAIL = 2;
    const T_API_STATUS = [
        self::T_API_STATUS_NO      => '— —',
        self::T_API_STATUS_SUCCESS => '转移成功',
        self::T_API_STATUS_FAIL    => '转移失败'
    ];
    // 取消订单
    const OP_CANCEL_ORDER = 'cancel_order';
    // 自主下架
    const OP_CLOSE_TRADE = 'close_trade';
    // 后台下架
    const OP_ADMIN_CLOSE_TRADE = 'admin_close_trade';
    /**
     *  跳转详情地址
     */
    const JUMP_DETAILS = [
        self::TYPE_GOLD => "/trade/gold/%d.html?type=0&mode=1",
        self::TYPE_ROLES => "/trade/role/%d.html?type=1&mode=1",
        self::TYPE_PROP => "/trade/prop/%d.html?type=2&mode=1",
    ];
    /**
     *  获取交易记录
     * @param $syncTradeno
     * @return array
     * @throws \think\db\exception\DataNotFoundException
     * @throws \think\db\exception\DbException
     * @throws \think\db\exception\ModelNotFoundException
     */
    public function getDetailsBySyncTradeno($syncTradeno = "")
    {
        $data = $this->where('sync_tradeno', $syncTradeno)->find();
        if (!$data) {
            return [];
        }

        return $data->toArray();
    }
    /**
     *  获取详情
     * @param $data
     * @param $fields
     * @return array
     */
    public function getDetailsByWhere($where = [], $fields = ['*'])
    {
        $details = self::field($fields)
            ->where($where)
            ->find();
        if (!$details) {
            return [];
        }

        return $details->toArray();
    }
    /**
     * 获取交易记录
     * @param string $game
     * @param int $serverid
     * @param $id
     * @return array
     * @throws \think\db\exception\DataNotFoundException
     * @throws \think\db\exception\DbException
     * @throws \think\db\exception\ModelNotFoundException
     */
    public function getDetailsById(string $game = "", int $serverid = 0, $id = 0)
    {
        $data = $this->field(["id", "game", "serverid", "userid", "username", "type", "sync_status", "sync_tradeno"])
            ->where('id', $id)
            ->find();
        if (!$data) {
            return [];
        }

        return $data->toArray();
    }

    /**
     *  列表
     * @param $data
     * @param $order
     * @param $limit
     * @return array
     */
    public function listIndexQuery($data = [], $order=['id'=>'desc'], $limit = 8)
    {
        $d        = request()->get();
        $game     = $data['game'] ?? "";
        $serverid = $data['serverid'] ?? 0;
        $username = $data['username'] ?? "";

        $where = [
            'game'     => $game,
            'serverid' => $serverid,
            'username'   => $username,
            ['created_at', '>=', date("Y-m-d H:i:s", (time() - config('app.seller_lists_time')))]
        ];

        // 排序
        $key   = $d['key'] ?? '';
        $value = $d['value'] ?? '';
        if (in_array($key, ['price', 'outtime']) && $key && $value) {
            $order = [$key => $value];
        }

        $fields = ['id', 'type'];
        $rs     = self::field($fields)
            ->where($where)
            ->order($order)
            ->field($fields)
            ->paginate([
                'query'     => $d,
                'var_page'  => 'p',
                'list_rows' => $limit,
            ]);
        $page   = $rs->render();
        $data   = $rs->getCollection()->toArray();

        $lists = [];
        if ($data) {
            $tids = array_column($data, 'type', 'id');

            // 按照类型区分id
            $tidLists = [];
            foreach ($tids as $k => $v) {
                $tidLists[$v][] = $k;
            }

            // 金币信息
            $goldLists = (new TradeGold())->getGoldListsByTids($game, $serverid, $tidLists[self::TYPE_GOLD] ?? []);
            $goldLists = array_column($goldLists, null, 'tid');

            // 角色信息
            $roleLists = (new TradeRole())->getRoleListsByTids($game, $serverid, $tidLists[self::TYPE_ROLES] ?? []);
            $roleLists = array_column($roleLists, null, 'tid');

            // 道具信息
            $propLists = (new TradeProp())->getPropListsByTids($game, $serverid, $tidLists[self::TYPE_PROP] ?? []);
            $propLists = array_column($propLists, null, 'tid');

            // 数据处理
            foreach ($data as $value) {
                switch ($value['type']) {
                    case self::TYPE_GOLD:
                        $item = $goldLists[$value['id']] ?? [];
                        break;
                    case self::TYPE_ROLES:
                        $item = $roleLists[$value['id']] ?? [];
                        break;
                    case self::TYPE_PROP:
                        $item = $propLists[$value['id']] ?? [];
                        break;
                }

                if (!$item) {
                    continue;
                }

                $lists[] = [
                    'tid'               => $value['id'],
                    'id'                => $item['id'],
                    'type'              => $value['type'],
                    'name'              => $item['name'] ?? "",
                    'og_text'           => $item['og_text'] ?? "",
                    'iconx'             => $item['iconx'] ?? 0,
                    'icony'             => $item['icony'] ?? 0,
                    'price'             => $item['price'] ?? 0,
                    'job_head'          => $item['job_head'] ?? "",
                    'iconset'           => $item['iconset'] ?? "",
                    'num'               => $item['num'] ?? 1,
                    'trade_status'      => $item['trade_status'] ?? 0,
                    'created_at'        => $item['created_at'] ?? "",
                    'outtime_type'      => $item['outtime_type'] ?? 0,
                    'outtime_text'      => $item['outtime_text'] ?? "",
                    'order_outtime_type'=> $item['order_outtime_type'] ?? 0,
                    'order_outtime_text'=> $item['order_outtime_text'] ?? "",
                    'trade_status_text' => $item['trade_status_text'] ?? "",
                ];
            }

            unset($data, $tidLists, $goldLists, $roleLists, $propLists);
        }

        return [
            'lists' => $lists,
            'page'  => $page,
        ];
    }

    public function TradeRole()
    {
        return $this->hasOne(TradeRole::class, 'tid', 'id');
    }
    public function TradeGold()
    {
        return $this->hasOne(TradeGold::class, 'tid', 'id');
    }
    public function TradeProp()
    {
        return $this->hasOne(TradeProp::class, 'tid', 'id');
    }
    /**
     * 动态获取关联关系
     * // 交易类型：0-金币 1-角色 2-道具
     */
    public static function getTypeTradeModel($type)
    {
        switch ($type) {
            case self::TYPE_GOLD: return 'TradeGold';
            case self::TYPE_ROLES: return 'TradeRole';
            case self::TYPE_PROP: return 'TradeProp';
            default: throw new \Exception('未知的交易类型');
        }
    }
    public static function getDynamicTrade($type)
    {
        $relationName = self::getTypeTradeModel($type);
        return self::with($relationName);
    }

    /**
     *  获取交易状态
     * @param int $tradeStatus
     * @param string $createdAt
     * @param int $orderPayStatus
     * @return array
     */
    public function getTradeStatus(int $tradeStatus = 0, string $createdAt = "", int $orderPayStatus = 0): array
    {
        $time       = time();
        $pubTime    = config('app.trade_public_time');
        $sellerTime = config('app.trade_seller_time');
        $totalTime  = (int)bcadd($pubTime, $sellerTime);
        $createdAt  = strtotime($createdAt);

        // 已下架、被下单、已出售
        if (in_array($tradeStatus, [
                self::T_TRADE_STATUS_PLACE,self::T_TRADE_STATUS_DONE,
                self::T_TRADE_STATUS_DOWN,self::T_TRADE_STATUS_SYSTEM_DOWN,self::T_TRADE_STATUS_ADMIN_DOWN,
            ])
        ) {
            return [
                $tradeStatus => self::T_TRADE_STATUS[$tradeStatus] ?? "",
            ];
        }

        // 无订单
        if ($orderPayStatus == Trade::TYPE_PAY_STATUS_NO) {
            if ($time > ($createdAt + $totalTime)) {
                return [self::T_TRADE_STATUS_SYSTEM_DOWN => self::T_TRADE_STATUS[self::T_TRADE_STATUS_SYSTEM_DOWN] ?? "",];
            }
        }

        // 公示期
        if ($createdAt >= ($time - $pubTime)) {
            return [self::T_TRADE_STATUS_PUB => self::T_TRADE_STATUS[self::T_TRADE_STATUS_PUB] ?? "",];
        }

        // 出售期
        if (($createdAt >= ($time - $totalTime)) && ($createdAt <= ($time - $pubTime))) {
            return [self::T_TRADE_STATUS_SELL => self::T_TRADE_STATUS[self::T_TRADE_STATUS_SELL] ?? "",];
        }

//         有订单
//        if ($orderCreatedAt) {
//            // 订单状态: 未超时+未支付=>被下单
//            if (in_array($orderPayStatus, [
//                    TradeOrder::TO_PAY_STATUS_TODO, TradeOrder::TO_PAY_STATUS_CACEL
//                ]) &&
//                $orderCreatedAt >= ($time - config('app.order_outtime'))
//            ) {
//                return [self::T_TRADE_STATUS_PLACE => self::T_TRADE_STATUS[self::T_TRADE_STATUS_PLACE] ?? "",];
//            }
//
//            // 订单状态：已超时+未支付+出售期=> 出售期
//            if (in_array($orderPayStatus, [
//                    TradeOrder::TO_PAY_STATUS_TODO, TradeOrder::TO_PAY_STATUS_CACEL
//                ]) &&
//                $orderCreatedAt < ($time - config('app.order_outtime')) &&
//                ($createdAt >= ($time - $totalTime)) &&
//                ($createdAt <= ($time - $pubTime))
//            ) {
//                return [self::T_TRADE_STATUS_SELL => self::T_TRADE_STATUS[self::T_TRADE_STATUS_SELL] ?? "",];
//            }
//        }
        // 订单状态：已超时+已支付 =》 状态回调中修改 => 不做处理
        // 订单状态：未超时+已支付 =》 状态回调中修改 => 不做处理

        return [
            $tradeStatus => self::T_TRADE_STATUS[$tradeStatus] ?? "",
        ];
    }

    /**
     * 关闭交易
     * @param $data
     */
    public function closeTrade($data = [])
    {
        $handleType = $data['handle_type'] ?? '';
        $model    = TradeModel::getTypeTradeModel($data['type']);
        $modelObj = app('app\\common\\model\\' . $model);

        if ($handleType == Trade::OP_CANCEL_ORDER) { // 取消订单-交易到期
            $tWhere = [
                'id'           => $data['id'],
                'trade_status' => Trade::T_TRADE_STATUS_PLACE,
            ];

            $tData = [
                'trade_status' => $data['trade_status'],
                'pay_status'   => Trade::TYPE_PAY_STATUS_NO,
                'orderno'      => ''
            ];
        } elseif (in_array($handleType,[Trade::OP_CLOSE_TRADE, Trade::OP_ADMIN_CLOSE_TRADE])) {
            $tWhere = [
                'id'           => $data['id'],
            ];

            $tData = [
                'trade_status' => $data['trade_status']
            ];
        } else { // 交易到期
            $tWhere = [
                'id'           => $data['id'],
                'trade_status' => Trade::T_TRADE_STATUS_SELL,
            ];

            $tData = [
                'trade_status' => $data['trade_status']
            ];
        }

        $sRes = $modelObj->where($tWhere)->update($tData);
        if (false === $sRes) {
            throw new Exception("更新数据失败");
        }

        // 站内信
        $noticeId = (new TradeNotice())->addNotice(
            $data['game'], $data['serverid'], $data['tid'], $data['username'], $data['orderno'] ?? "", $data['type'], $data['notice_type']
        );
        if (!$noticeId) {
            throw new Exception("添加站内信失败");
        }

        $res = GameTradeService::treasureremove($data['game'], $data['serverid'], $data['sync_tradeno']);
        if (($res['Success'] ?? 0) != 1) {
            throw new Exception("下架失败");
        }
    }
}