<?php

namespace app\common\command;

use app\common\model\TradeOrder;
use app\common\model\TradeStats;
use think\console\Command;
use think\console\Input;
use think\console\Output;
use think\Exception;
use think\facade\Log;
use think\facade\Db;

/**
 *  统计昨日交易数据
 */
class StatisticTradeData extends Command
{
    protected function configure()
    {
        $this->setName('StatisticTradeData')
            ->setDescription('交易数据统计');
    }

    protected function execute(Input $input, Output $output)
    {
        $output->info(date("Y-m-d H:i:s") . "交易数据统计-脚本开始执行");

        $time          = time();
        $yesterDayTime = $time - 86400;
        $yesterDayDate = date("Y-m-d", $yesterDayTime);
        $startTime     = strtotime(date("Y-m-d 00:00:00", $yesterDayTime));
        $endTime       = strtotime(date("Y-m-d 23:59:59", $yesterDayTime));

        try {

            $data = TradeOrder::field([
                "game", "serverid", "ordertype",
                Db::raw("SUM(price) as total_price"),
                Db::raw("SUM(charge) as total_charge"),
                Db::raw("SUM(CASE WHEN withdraw_status = 2 THEN amount ELSE 0 END) as total_withdraw"),
            ])
                ->where([
                    'pay_status' => TradeOrder::TO_PAY_STATUS_DONE,
                    ["paytime", "BETWEEN", [$startTime, $endTime]]
                ])
                ->group("game,serverid,ordertype")
                ->select()
                ->toArray();
            if (!$data) {
                return false;
            }

            $aData         = [];
            $totalPrice    = 0;
            $totalCharge   = 0;
            $totalWithdraw = 0;
            foreach ($data as $item) {
                $aData[] = [
                    'game'           => $item['game'],
                    'serverid'       => $item['serverid'],
                    'type'           => 0,
                    'ordertype'      => $item['ordertype'],
                    'year'           => date('Y', $yesterDayTime),
                    'month'          => date('m', $yesterDayTime),
                    'day'            => date('d', $yesterDayTime),
                    'total_price'    => $item['total_price'],
                    'total_charge'   => $item['total_charge'],
                    'total_withdraw' => $item['total_withdraw'],
                    'stats_date'     => $yesterDayDate,
                ];

                $totalPrice    = bcadd($totalPrice, $item['total_price'], 2);
                $totalCharge   = bcadd($totalCharge, $item['total_charge'], 2);
                $totalWithdraw = bcadd($totalWithdraw, $item['total_withdraw'], 2);
            }

            if ($aData) {
                // insertAll 字段顺序要一致，要和上面的字段顺序一样
                $aData[] = [
                    'game'           => $item['game'],
                    'serverid'       => $item['serverid'],
                    'type'           => 0,
                    'ordertype'      => -1, // 当前总数据
                    'year'           => date('Y', $yesterDayTime),
                    'month'          => date('m', $yesterDayTime),
                    'day'            => date('d', $yesterDayTime),
                    'total_price'    => $totalPrice,
                    'total_charge'   => $totalCharge,
                    'total_withdraw' => $totalWithdraw,
                    'stats_date'     => $yesterDayDate,
                ];
            }

            Db::startTrans();
            try {
                // 删除当天数据
                TradeStats::where('stats_date', $yesterDayDate)->delete();
                // 添加当天数据
                TradeStats::insertAll($aData);
                Db::commit();
            } catch (Exception $e) {
                Db::rollback();
                throw $e;
            }
        } catch (Exception $e) {
            $output->error(date("Y-m-d H:i:s") . "交易数据统计-脚本执行出错：" . $e->getMessage() . ", Line:" . $e->getLine() . ", File:" . $e->getFile());
            Log::channel("command")->info("交易数据统计-处理出错：" . json_encode([
                    'Message' => $e->getMessage(),
                    'Line'    => $e->getLine(),
                    'File'    => $e->getFile(),
                    'Trace'   => $e->getTraceAsString(),
                ], JSON_UNESCAPED_UNICODE));
        }
        $output->info(date("Y-m-d H:i:s") . "交易数据统计-脚本开始结束");
    }
}