<?php
namespace app\common\command;

use app\common\constant\SyncItem;
use app\common\model\Trade;
use app\common\model\TradeBackPack;
use app\common\model\TradeEquip;
use app\common\model\TradeGod;
use app\common\model\TradeRole;
use app\common\model\TradeWar;
use app\common\model\TradeWing;
use app\common\service\GameTradeService;
use think\console\Command;
use think\console\Input;
use think\console\input\Argument;
use think\console\Output;
use think\Exception;
use think\facade\Db;
use think\facade\Log;
use Throwable;

/**
 *  同步交易信息
 */
class PullTradeInfo extends Command
{
    protected function configure()
    {
        $this->setName('PullTradeInfo')
            ->addArgument('type', Argument::OPTIONAL, "执行类型 0-金币 1-角色 2-道具")
            ->addArgument('limit', Argument::OPTIONAL, "执行数量")
            ->setDescription('拉取交易信息数据');
    }

    /**
     *  状态：0-默认状态 1-失败 2-成功
     */
    const STATUS_DEFAULT = 0;
    const STATUS_FAILURE = 1;
    const STATUS_SUCCESS = 2;

    /**
     *  同步数据流程
     * @var array
     */
    protected $typeProcessConfig = [];

    /**
     *  拉取交易信息
     * @param Input $input
     * @param Output $output
     * @return void
     */
    protected function execute(Input $input, Output $output)
    {
        $output->info(date("Y-m-d H:i:s") . "数据同步-脚本开始执行");
        try {
            $limit = (int)$input->getArgument('limit');
            $type  = (int)$input->getArgument('type');
            $limit = $limit ?: 20;

            $where = [
                'type' => $type,
                ['sync_status', "IN", [Trade::T_SYNC_STATUS_TODO, Trade::T_SYNC_STATUS_FAIL]]
            ];

            // 延时获取角色数据
            if ($type === Trade::TYPE_ROLES) {
                $where[] = ['created_at', '<=', date("Y-m-d H:i:s", (time() - config('app.role_delayed_time')))];
            }

            // 获取同步失败和待同步的交易数据
            $tradeLists = $this->getTradeLists($where, $limit);
            if (!$tradeLists) {
//                $output->info(date("Y-m-d H:i:s") . "数据同步-没有可执行的数据");
                return;
            }

            $roleTids        = $this->getRoleTradeIds($tradeLists);
            $roleCharidLists = $this->getRoleCharidLists($roleTids);

            foreach ($tradeLists as $item) {
                $item['charid'] = $roleCharidLists[$item['id']] ?? 0;
                $this->typeProcessConfig = config($item['game'] . ".sync_item_flow");
                $res            = $this->processTradeItem($item);
                if ($res["status"] === false) {
                    throw new Exception($res["msg"] ?? "系统错误");
                }
            }
        } catch (Throwable $e) {
            $output->info(date("Y-m-d H:i:s") . "数据同步-脚本执行出错：" . $e->getMessage() . ", Line:" . $e->getLine() . ", File:" . $e->getFile());
            Log::channel("command")->info("数据同步-脚本执行出错" . json_encode([
                    'Message' => $e->getMessage(),
                    'Line'    => $e->getLine(),
                    'File'    => $e->getFile(),
                    'Trace'   => $e->getTraceAsString(),
                ], JSON_UNESCAPED_UNICODE));
        }

        $output->info(date("Y-m-d H:i:s") . "数据同步-脚本结束执行");
    }

    /**
     * 获取数据
     * @param $item
     * @return array
     */
    private function getTradeData($item = []): array
    {
        $data = [];
        $stepLists = $this->getStepStatus($item['serverid'], $item['type'], $item["step_status"]);
        foreach ($stepLists as $step => $value) {
            // 已经获取成功的不在获取
            if ($value["status"] == self::STATUS_SUCCESS) {
                // 不可删除
                $data[$step] = [
                    "status" => self::STATUS_SUCCESS,
                    "error"  => ""
                ];
                continue;
            }
            $method = snakeToMethod($step);
            if (method_exists($this, $method)) {
                $result      = $this->$method($item);
                $data[$step] = [
                    'status' => $result['status'] ?? self::STATUS_DEFAULT,
                    'error'  => $result['msg'] ?? '',
                    'data'   => $result['data'] ?? []
                ];

                // 更新charid用于后续步骤
                if ($step === SyncItem::STEP_UPPER && isset($result['data']['chaid'])) {
                    $item['charid'] = $result['data']['chaid'];
                }
            }
        }
        return $data;
    }

    /**
     *  交易状态
     * @param $time
     * @return int|mixed
     */
    public function getTradeStatus($time = 0)
    {
        $timeText    = calculateRemainingTimeText($time, time());
        $tradeStatus = [
            1 => Trade::T_TRADE_STATUS_SELL,
            2 => Trade::T_TRADE_STATUS_PUB,
            3 => Trade::T_TRADE_STATUS_SYSTEM_DOWN,
        ];
        return $tradeStatus[key($timeText)] ?? 0;
    }

    /**
     *  获取可执行数据列表
     *
     * @param array $where
     * @param int $limit
     * @return array
     */
    private function getTradeLists(array $where = [], int $limit = 20): array
    {
        $data = Trade::field(["id", "game", "serverid", "username", "type", "sync_tradeno", "sync_status", "step_status", "created_at"])
            ->where($where)
            ->limit($limit)
            ->order("sync_status", "desc")
            ->select()
            ->toArray();

        if ($data) {
            foreach ($data as &$value) {
                if (!$value['step_status']) {
                    $value['step_status'] = [];
                } else {
                    $value['step_status'] = json_decode($value['step_status'], true);
                }
            }
        }

        return $data;
    }

    /**
     *  获取ids
     * @param array $tradeLists
     * @return array
     */
    private function getRoleTradeIds(array $tradeLists = []): array
    {
        // 查询已经获取的上架信息 目的批量获取chaid
        $roleTids = array_column(array_filter($tradeLists, function ($trade) {
            $upperStatus = $trade['step_status']['upper']['status'] ?? false;
            return $trade['type'] == Trade::TYPE_ROLES && $upperStatus == self::STATUS_SUCCESS;
        }), "id");

        return $roleTids;
    }

    /**
     *  获取已有数据
     * @param array $tids
     * @return array
     */
    public function getRoleCharidLists(array $tids = []): array
    {
        if (!$tids) {
            return [];
        }

        return TradeRole::whereIn("tid", $tids)->column("chaid", "tid");
    }

    /**
     *  处理数据
     * @param $item
     * @return array
     */
    public function processTradeItem($item)
    {
        $tradeDataLists = $this->getTradeData($item);
        if (empty($tradeDataLists)) {
            return [
                "status" => false,
                "msg"    => "未获取到数据"
            ];
        }

        Db::startTrans();
        try {
            $res = $this->processTradeData($item, $tradeDataLists);
            $this->updateTradeStatus($item['id'], $item['type'], $item['serverid'],$res);
            Db::commit();
        } catch (Throwable $e) {
            Db::rollback();
            throw $e;
        }

        return [
            "status" => true,
            "msg"    => "获取成功",
        ];
    }

    /**
     * 处理交易数据
     */
    protected function processTradeData(array $item, array $tradeDataLists): array
    {
        $sData = [];
        foreach ($tradeDataLists as $stepName => $stepData) {

            $data = $stepData['data'] ?? [];
            unset($tradeDataLists[$stepName]['data']);
            if (!$data) {
                continue;
            }

            switch ($stepName) {
                case SyncItem::STEP_UPPER:
                    $sData = $this->processUpperStep($item, $data, $sData);
                    break;
                case SyncItem::STEP_DETAILS:
                    $this->processDetailsStep($data);
                    break;
                case SyncItem::STEP_PACKSACK:
                case SyncItem::STEP_ROLE_BACKPACK:
                case SyncItem::STEP_FACADE:
                    $this->processBackpackStep($data);
                    break;
                case SyncItem::STEP_ROLE_GOD:
                    $this->processGodStep($data);
                    break;
                case SyncItem::STEP_ROLE_WING:
                    $this->processWingStep($data);
                    break;

            }
        }
        return [
            'sData'          => $sData,
            'tradeDataLists' => $tradeDataLists
        ];
    }

    /**
     * 处理上架信息步骤
     */
    protected function processUpperStep(array $item, array $data, array $sData): array
    {
        $sData['price']   = $data['price'] ?? 0;
        $sData['outtime'] = $data['outtime'] ?? 0;

        if ($data) {
            $model = Trade::getTypeTradeModel($item['type']);
            app('app\\common\\model\\' . $model)->insert($data);
        }

        return $sData;
    }

    /**
     * 处理详情步骤
     */
    protected function processDetailsStep(array $data): void
    {
        if (!empty($data[SyncItem::STEP_ROLE_WAR])) {
            TradeWar::insertAll($data[SyncItem::STEP_ROLE_WAR]);
        }

        if (!empty($data[SyncItem::STEP_ROLE_EQUIP])) {
            TradeEquip::insertAll($data[SyncItem::STEP_ROLE_EQUIP]);
        }
    }

    /**
     * 处理背包步骤
     */
    protected function processBackpackStep(array $data): void
    {
        if ($data) {
            TradeBackPack::insertAll($data);
        }
    }

    /**
     * 处理天神步骤
     */
    protected function processGodStep(array $data): void
    {
        if ($data) {
            TradeGod::insertAll($data);
        }
    }

    /**
     * 处理翅膀步骤
     */
    protected function processWingStep(array $data): void
    {
        if ($data) {
            TradeWing::insertAll($data);
        }
    }

    /**
     * 更新交易状态
     */
    protected function updateTradeStatus(int $tradeId = 0, int $type = 0, int $serverid = 0,array $data = []): void
    {
        $sData                = $data['sData'] ?? [];
        $tradeResLists        = $data['tradeDataLists'] ?? [];
        $syncStatus           = $this->getSyncStatus($serverid, $type, array_column($tradeResLists, 'status'));
        $sData['sync_status'] = $syncStatus;
        $sData['step_status'] = json_encode($tradeResLists, JSON_UNESCAPED_UNICODE);
        Trade::where('id', $tradeId)->update($sData);
    }
    /**
     * 获取上架信息数据
     */
    protected function getUpperData(array $item): array
    {
        if (empty($item['sync_tradeno'])) {
            return [
                "status" => self::STATUS_DEFAULT,
                "msg"    => "交易流水号为空",
                "data"   => [],
            ];
        }

        $game       = $item["game"] ?? "";
        $serverid   = $item["serverid"] ?? 0;
        $res = GameTradeService::getTreasureListed($game, $serverid, $item['sync_tradeno']);
        if (false === ($res['status'] ?? false) || !$res) {
            return [
                "status" => self::STATUS_FAILURE,
                "msg"    => $res["msg"] ?? "系统错误",
                "data"   => [],
            ];
        }

        return [
            "status" => self::STATUS_SUCCESS,
            "msg"    => "",
            "data"   => $this->formatUpperData($item, $res['data'] ?? [])
        ];
    }
    /**
     * 格式化上架数据
     */
    protected function formatUpperData(array $item, array $res): array
    {
        if (!$res) {
            return [];
        }

        $game       = $item["game"] ?? "";
        $time       = strtotime($item["created_at"]);
        $commonData = [
            "tid"          => $item["id"] ?? 0,
            "game"         => $game,
            "serverid"     => $item["serverid"] ?? 0,
            "username"     => $item["username"] ?? "",
            "specuser"     => $res["appoint"] ?? "",
            "price"        => $res["value"] ?? 0,
            "trade_status" => $this->getTradeStatus($time),
            "outtime"      => ($time + tradeOuttime()),
            "created_at"   => $item['created_at'],
            "chaid"        => $res["charid"] ?? 0,
            "chaname"      => $res["charname"] ?? "",
            "chalv"        => $res["level"] ?? 0,
            "family"       => $res["family"] ?? 0,
            "grow"         => $res["grow"] ?? 0,
            "job"          => $res["job"] ?? 0,
        ];

        switch ($item['type']) {
            case Trade::TYPE_GOLD:
                $num       = bcmul(($res["number"] ?? 0), config($game . '.gold_num_ratio'));
                $unitPrice = $num ? bcdiv(($res["value"] ?? 0), $num, 2) : 0;
                return array_merge($commonData, [
                    "num"        => $num,
                    "unit_price" => $unitPrice,
                ]);
            case Trade::TYPE_ROLES:
                return array_merge($commonData, [
                    "title" => $res["desc"] ?? "",
                    "kill"  => $res["kill"] ?? 0,
                    "money" => $res["money"] ?? 0,
                ]);
            case Trade::TYPE_PROP:
                return array_merge($commonData, [
                    "num"        => $res["number"] ?? 0,
                    "itemid"     => $res["itemid"] ?? 0,
                    "unit_price" => $res['number'] ? bcdiv(($res["value"] ?? 0), ($res['number'] ?? 1), 2) : 0,
                ]);
            default:
                return $commonData;
        }
    }
    /**
     * 获取详情数据
     */
    protected function getDetailsData(array $item): array
    {
        if (empty($item['sync_tradeno'])) {
            return [
                "status" => self::STATUS_DEFAULT,
                "msg"    => "交易流水号为空",
                "data"   => [],
            ];
        }

        $game    = $item["game"] ?? "";
        $serverid   = $item["serverid"] ?? 0;
        $res = GameTradeService::getTreasureDetail($game, $serverid, $item['sync_tradeno']);
        if (false === ($res['status'] ?? false) || !$res) {
            return [
                "status" => self::STATUS_FAILURE,
                "msg"    => $res["msg"] ?? "系统错误",
                "data"   => [],
            ];
        }
        return [
            "status" => self::STATUS_SUCCESS,
            "msg"    => "",
            "data"   => $this->formatDetailsData($item, $res['data'] ?? [])
        ];
    }

    /**
     * 格式化详情数据
     */
    protected function formatDetailsData(array $item, array $res): array
    {
        $time  = time();
        $war   = $res['data']['war'] ?? [];
        $equip = $res['data']['equip'] ?? [];

        return [
            SyncItem::STEP_ROLE_WAR   => $this->formatWarData($item, $war, $time),
            SyncItem::STEP_ROLE_EQUIP => $this->formatEquipData($item, $equip, $time)
        ];
    }

    /**
     * 格式化战斗数据
     */
    protected function formatWarData(array $item, array $war, int $time): array
    {
        if (!$war) {
            return [];
        }

        $warLists = [];
        foreach ($war as $v) {
            $warLists[] = [
                "tid"        => $item["id"] ?? 0,
                "game"       => $item["game"] ?? "",
                "serverid"   => $item["serverid"] ?? 0,
                "type"       => $v["war"] ?? 0,
                "hp"         => $v["hp"] ?? 0,
                "sp"         => $v["sp"] ?? 0,
                "mp"         => $v["mp"] ?? 0,
                "str"        => $v["str"] ?? 0,
                "dex"        => $v["dex"] ?? 0,
                "int"        => $v["int"] ?? 0,
                "wis"        => $v["wis"] ?? 0,
                "rec"        => $v["rec"] ?? 0,
                "luc"        => $v["luc"] ?? 0,
                "ap"         => $v["ap"] ?? 0,
                "dp"         => $v["dp"] ?? 0,
                "af"         => $v["af"] ?? 0,
                "as"         => $v["as"] ?? 0,
                "am"         => $v["am"] ?? 0,
                "df"         => $v["df"] ?? 0,
                "ds"         => $v["ds"] ?? 0,
                "dm"         => $v["dm"] ?? 0,
                "created_at" => date("Y-m-d H:i:s", $time),
            ];
        }
        return $warLists;
    }

    /**
     * 格式化装备数据
     */
    protected function formatEquipData(array $item, array $equip, int $time): array
    {
        if (!$equip) {
            return [];
        }
        $equipLists = [];
        foreach ($equip as $vv) {
            foreach ($vv['items'] as $vvv) {
                $equipLists[] = [
                    "tid"        => $item["id"] ?? 0,
                    "game"       => $item["game"] ?? "",
                    "serverid"   => $item["serverid"] ?? 0,
                    "type"       => $vv["war"] ?? 0,
                    "itemid"     => $vvv["id"] ?? 0,
                    "st"         => $vvv["st"] ?? 0,
                    "hp"         => $vvv["hp"] ?? 0,
                    "sp"         => $vvv["sp"] ?? 0,
                    "mp"         => $vvv["mp"] ?? 0,
                    "str"        => $vvv["str"] ?? 0,
                    "dex"        => $vvv["dex"] ?? 0,
                    "int"        => $vvv["int"] ?? 0,
                    "wis"        => $vvv["wis"] ?? 0,
                    "rec"        => $vvv["rec"] ?? 0,
                    "luc"        => $vvv["luc"] ?? 0,
                    "ia"         => $vvv["ia"] ?? 0,
                    "iav"        => $vvv["iav"] ?? 0,
                    "ide"        => $vvv["ide"] ?? 0,
                    "idm"        => $vvv["idm"] ?? 0,
                    "rd"         => $vvv["rd"] ?? 0,
                    "sl"         => $vvv["sl"] ?? 0,
                    "g1"         => $vvv["g1"] ?? 0,
                    "g2"         => $vvv["g2"] ?? 0,
                    "g3"         => $vvv["g3"] ?? 0,
                    "g4"         => $vvv["g4"] ?? 0,
                    "g5"         => $vvv["g5"] ?? 0,
                    "g6"         => $vvv["g6"] ?? 0,
                    "cj"         => $vvv["cj"] ?? 0,
                    "created_at" => date("Y-m-d H:i:s", $time),
                ];
            }
        }
        return $equipLists;
    }
    /**
     * 获取背包数据
     */
    protected function getPacksackData(array $item): array
    {
        return $this->getBackpackData($item, TradeBackPack::TYPE_PACKSACK);
    }

    /**
     * 获取角色仓库数据
     */
    protected function getRoleBackpackData(array $item): array
    {
        return $this->getBackpackData($item, TradeBackPack::TYPE_ROLE_BACKPACK);
    }

    /**
     * 获取背包通用数据
     */
    protected function getBackpackData(array $item, int $type): array
    {
        if (empty($item['charid'])) {
            return [
                "status" => self::STATUS_DEFAULT,
                "msg"    => "角色ID为空",
                "data"   => [],
            ];
        }

        $game    = $item["game"] ?? "";
        $serverid = $item["serverid"] ?? 0;
        $res = GameTradeService::getCharbackpack($game, $serverid, $item['charid'], $type);
        if (false === ($res['status'] ?? false) || !$res) {
            return [
                "status" => self::STATUS_FAILURE,
                "msg"    => $res["msg"] ?? "系统错误",
                "data"   => [],
            ];
        }

        return [
            "status" => self::STATUS_SUCCESS,
            "msg"    => "",
            "data"   => $this->formatBackpackData($item, $res['data'] ?? [], $type)
        ];
    }

    /**
     * 格式化背包数据
     */
    protected function formatBackpackData(array $item, array $itemData, int $type): array
    {
        if (!$itemData) {
            return [];
        }
        $time = time();
        $data = [];
        foreach ($itemData as $v) {
            $data[] = [
                "tid"        => $item["id"] ?? 0,
                "game"       => $item["game"] ?? "",
                "serverid"   => $item["serverid"] ?? 0,
                "type"       => $type,
                "itemid"     => $v['itemid'],
                "num"        => $v['count'],
                "created_at" => date("Y-m-d H:i:s", $time),
            ];
        }
        return $data;
    }

    /**
     * 获取外观数据
     */
    protected function getFacadeData(array $item): array
    {
        if (empty($item['charid'])) {
            return [
                "status" => self::STATUS_DEFAULT,
                "msg"    => "角色ID为空",
                "data"   => [],
            ];
        }

        $game    = $item["game"] ?? "";
        $serverid = $item["serverid"] ?? 0;
        $res = GameTradeService::getCharfashion($game, $serverid, $item['charid']);
        if (false === ($res['status'] ?? false) || !$res) {
            return [
                "status" => self::STATUS_FAILURE,
                "msg"    => $res["msg"] ?? "系统错误",
                "data"   => [],
            ];
        }

        return [
            "status" => self::STATUS_SUCCESS,
            "msg"    => "",
            "data"   => $this->formatFacadeData($item, $res['data'] ?? [])
        ];
    }

    /**
     * 格式化外观数据
     */
    protected function formatFacadeData(array $item, array $itemData): array
    {
        if (!$itemData) {
            return [];
        }
        $time = time();
        $data = [];
        foreach ($itemData as $v) {
            $data[] = [
                "tid"        => $item["id"] ?? 0,
                "game"       => $item["game"] ?? "",
                "serverid"   => $item["serverid"] ?? 0,
                "type"       => TradeBackPack::TYPE_FACADE,
                "itemid"     => $v['itemid'],
                "num"        => 0,
                "created_at" => date("Y-m-d H:i:s", $time),
            ];
        }
        return $data;
    }

    /**
     *  获取天神数据
     * @param $item
     * @return array
     */
    public function getGodData($item)
    {
        if (empty($item['charid'])) {
            return [
                "status" => self::STATUS_DEFAULT,
                "msg"    => "角色ID为空",
                "data"   => [],
            ];
        }

        $game    = $item["game"] ?? "";
        $serverid = $item["serverid"] ?? 0;
        $res = GameTradeService::getChargoddata($game, $serverid, $item['charid']);
        if (false === ($res['status'] ?? false) || !$res) {
            return [
                "status" => self::STATUS_FAILURE,
                "msg"    => $res["msg"] ?? "系统错误",
                "data"   => [],
            ];
        }

        return [
            "status" => self::STATUS_SUCCESS,
            "msg"    => "",
            "data"   => $this->formatGodData($item, $res['data'] ?? [])
        ];
    }

    /**
     *  天神数据
     * @param array $item
     * @param array $itemData
     * @return array
     */
    public function formatGodData(array $item, array $itemData): array
    {
        if (!$itemData) {
            return [];
        }
        $data = [];
        foreach ($itemData as $v) {
            $data[] = [
                "tid"      => $item["id"] ?? 0,
                "game"     => $item["game"] ?? "",
                "serverid" => $item["serverid"] ?? 0,
                "godid"    => $v['godid'],
                "index"    => $v['index'],
                "mul"      => $v['mul'],
                "level"    => $v['level'],
                "str"      => $v['str'],
                "dex"      => $v['dex'],
                "rec"      => $v['rec'],
                "int"      => $v['int'],
                "luc"      => $v['luc'],
                "wis"      => $v['wis'],
                "hp"       => $v['hp'],
                "mp"       => $v['mp'],
                "sp"       => $v['sp'],
                "at"       => $v['at'],
                "atv"      => $v['atv'],
                "df"       => $v['df'],
                "dfm"      => $v['dfm'],
            ];
        }
        return $data;
    }

    /**
     *  获取天神数据
     * @param $item
     * @return array
     */
    public function getWingData($item)
    {
        if (empty($item['charid'])) {
            return [
                "status" => self::STATUS_DEFAULT,
                "msg"    => "角色ID为空",
                "data"   => [],
            ];
        }

        $game    = $item["game"] ?? "";
        $serverid = $item["serverid"] ?? 0;
        $res = GameTradeService::getCharwingdata($game, $serverid, $item['charid']);
        if (false === ($res['status'] ?? false) || !$res) {
            return [
                "status" => self::STATUS_FAILURE,
                "msg"    => $res["msg"] ?? "系统错误",
                "data"   => [],
            ];
        }
        return [
            "status" => self::STATUS_SUCCESS,
            "msg"    => "",
            "data"   => $this->formatWingData($item, $res['data'] ?? [])
        ];
    }

    /**
     *  翅膀数据
     * @param array $item
     * @param array $itemData
     * @return array
     */
    public function formatWingData(array $item, array $itemData): array
    {
        if (!$itemData) {
            return [];
        }
        $data = [];
        foreach ($itemData as $v) {
            $data[] = [
                "tid"      => $item["id"] ?? 0,
                "game"     => $item["game"] ?? "",
                "serverid" => $item["serverid"] ?? 0,
                "wingid"   => $v['wingid'],
                "level"    => $v['level'],
                "grade"    => $v['grade'],
                "str"      => $v['str'],
                "dex"      => $v['dex'],
                "rec"      => $v['rec'],
                "int"      => $v['int'],
                "luc"      => $v['luc'],
                "wis"      => $v['wis'],
            ];
        }
        return $data;
    }

    /**
     * 获取执行情况
     * @param int $serverid
     * @param int $type
     * @param array $stepStatus
     * @return array
     */
    public function getStepStatus(int $serverid = 0, int $type = 0, array $stepStatus = []): array
    {
        if (!$stepStatus) {
            $stepStatus = [];
        }

        $defaultStatus = $this->setDefaultStepStatus($serverid, $type);
        return array_merge($defaultStatus, $stepStatus);
    }
    /**
     * 状态列表
     * @param int $serverid
     * @param int $type
     * @param array $statusLists
     * @return int
     */
    public function getSyncStatus(int $serverid = 0, int $type = 0, array $statusLists = []): int
    {
        // 更新状态
        $statusCount  = array_count_values($statusLists);
        $successCount = count($this->typeProcessConfig[$serverid][$type] ?? []);
        $syncStatus   = ($statusCount[self::STATUS_SUCCESS] ?? 0) == $successCount ? Trade::T_SYNC_STATUS_SUCCESS : Trade::T_SYNC_STATUS_FAIL;
        return $syncStatus;
    }

    /**
     * 设置默认数据
     * @param int $serverid
     * @param int $type
     * @return array
     */
    public function setDefaultStepStatus(int $serverid = 0, int $type = 0): array
    {
        $defaultStatus = $this->typeProcessConfig[$serverid][$type] ?? [];
        if (!$defaultStatus) {
            return [];
        }
        foreach ($defaultStatus as $k => $v) {
            unset($defaultStatus[$k]);
            $defaultStatus[$v] = ["status" => self::STATUS_DEFAULT, "error" => ""];
        }
        return $defaultStatus;
    }
}