<?php
namespace app\admin\controller\order;

use app\admin\controller\AdminBase;
use app\common\constant\CacheKey;
use app\common\constant\CommonKey;
use app\common\model\Trade;
use app\common\model\TradeGame;
use app\common\model\TradeOrder;
use app\common\model\TradeWithdraw;
use app\common\service\FileLockService;
use app\common\service\WithdrawTradeService;
use think\Exception;
use think\facade\Db;
use think\facade\Log;

class WithdrawTrade extends AdminBase
{
    public function index($do = '')
    {
        if ($do == 'json') {
            $list = (new TradeWithdraw())->listQuery();
            return $this->returnMsg($list);
        }

        $gameObj = new TradeGame();
        $game    = $gameObj->getGameLists();

        $this->assign([
            'limit'           => 10,
            'game'            => arrayToJson($game['game_lists'] ?? []),
            'serverid'        => arrayToJson($game['serverid_lists'] ?? []),
            'type'            => arrayToJson(Trade::TYPE),
            'withdraw_status' => arrayToJson(TradeWithdraw::WITHDRAW_STATUS),
        ]);
        return $this->fetch();
    }

    /**
     * 查看
     * @param $do
     * @return string
     */
    public function details($do = '')
    {
        $id      = $this->request->get('id');
        $details = (new TradeWithdraw())->getDetailsByWhere(['id' => $id]);
        $details['alitime_text'] = $details['alitime'] ? date('Y-m-d H:i:s', $details['alitime']) : '--';
        $details['withdrawtime_text'] = $details['withdrawtime'] ? date('Y-m-d H:i:s', $details['withdrawtime']) : '--';
        $gameObj = new TradeGame();
        $game    = $gameObj->getGameLists();
        $this->assign([
            'details'         => $details,
            'types'           => Trade::TYPE,
            'game'            => $game['game_lists'] ?? [],
            'serverid'        => $game['serverid_lists'] ?? [],
            'withdraw_status' => TradeWithdraw::WITHDRAW_STATUS,
        ]);
        return $this->fetch();
    }

    /**
     *  转账
     * @param string $do
     * @return string|\think\response\Json
     */
    public function payment(string $do = '')
    {
        $orderno = $this->request->post("orderno", "");
        if (!$orderno) {
            return $this->returnMsg("缺少参数");
        }

        try {
            $obj = new FileLockService();
            $lockKey = sprintf(CacheKey::ADMIN_WITHDRAW_PAYMENT_LOCK, $orderno);
            $fp = $obj->acquireLock($lockKey, CommonKey::EXPIRED_THREE_SECOND);
            if ($fp === false) {
                throw new Exception("操作频繁");
            }

            $withdrawObj = new TradeWithdraw();
            $orderObj    = new TradeOrder();
            $withdraw    = $withdrawObj->getDetailsByWhere([
                'orderno' => $orderno
            ], ['id', 'game', 'serverid', 'tid', 'relationid', 'ordertype', 'payuser', 'aliname', 'status', 'alitime', 'transfer_order_id']);
            if (!$withdraw) {
                throw new Exception("提现信息不存在，无法提现");
            }

            $withdrawId = $withdraw['id'];
            $payuser    = $withdraw['payuser'] ?? "";
            $aliname    = $withdraw['aliname'] ?? "";
            $status     = $withdraw['status'] ?? 0;
            if (!$payuser || !$aliname) {
                throw new Exception("提现参数错误，无法提现");
            }

            if ($status == TradeWithdraw::STATUS_SUCCESS) {
                throw new Exception("已提现，请勿重复操作");
            }

            // 获取订单信息
            $order = $orderObj->getDetailsByWhere([
                'orderno' => $orderno
            ], ['id', 'pay_status', 'api_status', 'amount']);
            if (!$order) {
                throw new Exception("订单数据不存在，无法提现");
            }

            $orderId     = $order['id'];
            $apiStatus   = $order['api_status'] ?? 0;
            $payStatus   = $order['pay_status'] ?? 0;

            if ($payStatus != TradeOrder::TO_PAY_STATUS_DONE) {
                throw new Exception("订单未支付成功，无法提现");
            }

            if ($apiStatus != TradeOrder::API_STATUS_SUCCESS) {
                throw new Exception("道具转移未成功，无法提现");
            }

            $amount = $order['amount'] ?? 0;
            if ($amount < 1) {
                throw new Exception("提现金额错误，无法提现");
            }

            Db::startTrans();
            try {
                // 提现信息状态
                TradeWithdraw::where('id', $withdrawId)->update(['status' => TradeWithdraw::STATUS_SUCCESS]);
                // 订单-提现状态
                TradeOrder::where('id', $orderId)->update(['withdraw_status' => TradeWithdraw::STATUS_SUCCESS]);
                // 更新订单提现状态
                $res = withdrawTradeService::withdrawTrade($orderno, $amount, $payuser, $aliname);
                if (false == ($res['status'] ?? false)) {
                    throw new Exception($res['msg'] ?? "系统错误");
                }
                Db::commit();
            } catch (Exception $ex) {
                Db::rollback();
                throw $ex;
            }

            // 更新提现信息
            if ($res['status']) {
                try {
                    $data         = $res['data'] ?? [];
                    $transDate    = $data['trans_date'] ?? "";
                    $transOrderId = $data['order_id'] ?? "";
                    TradeWithdraw::where('id', $withdrawId)
                        ->update([
                            'status'            => TradeWithdraw::STATUS_SUCCESS,
                            'withdrawtime'      => strtotime($transDate),
                            'transfer_order_id' => $transOrderId,
                        ]);
                } catch (Exception $exx) {
                    Log::error("提现成功，更新提现信息失败：" . json_encode([
                            'message' => $exx->getMessage(),
                            'code'    => $exx->getCode(),
                            'line'     => $exx->getLine(),
                            'file'     => $exx->getFile(),
                            'trace'    => $exx->getTraceAsString(),
                            'data'     => $data,
                        ], JSON_UNESCAPED_UNICODE));
                    throw new Exception("提现成功，更新提现信息失败");
                }
            }
        } catch (Exception $e) {
            return $this->returnMsg($e->getMessage());
        } finally {
            if ($fp !== false) {
                $obj->deleteLockFile($fp, $lockKey);
            }
        }

        return $this->returnMsg("提现成功", 1);
    }
}