<?php

declare (strict_types=1);
namespace app\admin;

use app\admin\model\system\Online;
use think\App;
use think\exception\ValidateException;
use think\Response;
use think\facade\View;
use think\exception\HttpResponseException;
use think\Validate;

/**
 * 控制器抽象基类
 */
abstract class BaseController
{
    /**
     * 应用实例
     * @var App
     */
    protected $app;

    /**
     * Request实例
     * @var \think\Request
     */
    protected $request;

    /**
     * 信息模板
     * @var string
     */
    protected $msgTpl = '';

    /**
     * 控制器中间件
     * @var array
     */
    protected $middleware = [];

    /**
     * 前台会员信息
     * @var array
     */
    protected $memUser = [];

    /**
     * 构造方法
     * @param App $app
     */
    public function __construct(App $app)
    {
        // 应用实例
        $this->app = $app;
        // 请求对象
        $this->request = $this->app->request;
        // 映射路径
        defined('APP_MAP') or define('APP_MAP', $this->request->root());
        // 验证（登录、权限）
        $this->__auth();
        // 控制器初始化
        $this->__init();
    }

    /**
     * 验证（登录、权限）
     */
    protected function __auth(){}

    /**
     * 初始化
     */
    protected function __init(){}

    /**
     * 空方法
     */
    public function __call($name, $arg)
    {
        $this->exitMsg('Method does not exist',400);
    }

    /**
     * 模板赋值
     * @param ...$vars
     * @return void
     */
    protected function assign(...$vars)
    {
        View::assign(...$vars);
    }
    /**
     * 模板渲染
     * @param string $tmp
     * @param string $tip
     * @return string
     */
    protected function fetch(string $tmp = '', string $tip = '')
    {
        $this->logon($tip);
        return View::fetch($tmp);
    }
    /**
     * 重定向
     * @access  protected
     * @param   string  $args  重定向地址
     * @throws  HttpResponseException
     */
    protected function redirect(...$args)
    {
        throw new HttpResponseException(redirect(...$args));
    }

    /**
     * 日志/在线处理
     * @param string $tip
     * @return void
     */
    protected function logon(string $tip = ''){
        $url = substr(vhtmlspecialchars(strip_sql($this->request->url())),0,200);
        $Online = $this->memUser ? ['userid'=>'m'.$this->memUser['userid'],'username'=>$this->memUser['username']] : session('VT_ONLINE');
        if(!$Online){
            $Online = ['userid'=>uniqid(),'username'=>'游客'];
            session('VT_ONLINE',$Online);
        }
        Online::create(['userid'=>$Online['userid'],'username'=>$Online['username'],'url'=>$url,'etime'=>ZBG_TIME,'ip'=>ZBG_IP,'type'=>1],['userid','username','url','etime','ip','type'],true);
    }


    /**
     * 中断反馈信息
     * @access  protected
     * @param   string   $m   信息字符
     * @param   int      $c   状态值 303网址请求未登录
     * @param   array    $d   数组信息
     * @param   array    $h   发送的Header信息
     * @throws  HttpResponseException
     */
    protected function exitMsg($m, $c = 0, $d = [], $h = [])
    {
        if($c == 303){
            $re = Response::create(ROOT_PATH . 'app/admin/error.tpl','view')->assign(['msg'=>$m,'url'=>$d['url']])->header($h);
        }else{
            $rs = json_encode(['code'=>$c,'msg'=>$m,'data'=>$d]);
            $re = Response::create($rs)->header($h);
        }
        throw new HttpResponseException($re);
    }
    protected function returnMsg($msg = '', $code = 0, $data = [], $scode = 200, $header = [], $options = [])
    {
        $msg = is_object($msg) ? $msg->toArray() : $msg;
        if(is_array($msg)){
            if(isset($msg['total'])){ //分页模式
                $data  = $msg['data'];
                $count = $msg['total'];
                $data['msg'] = $msg['msg'] ?? '';
            }else{
                $data = $msg;
            }
            $msg  = $data['msg'] ?? ''; unset($data['msg']);
            $code = $data['code'] ?? $code; unset($data['code']);
            $data = $data['data'] ?? $data;
        }elseif(!$this->msgTpl){
            $this->logon($msg);
        }
        $count = isset($count) ? $count : (is_array($data) ? count($data) : 1);
        if($this->msgTpl){
            $this->assign(compact('code', 'msg', 'data', 'count'));
            return $this->fetch($this->msgTpl);
        }else{
            return json(compact('code', 'msg', 'data', 'count'), $scode, $header, $options);
        }
    }
    /**
     * 带模板反馈提示
     * @access   protected
     * @param    string    $msg    提示信息
     * @param    int       $tpl    提示模板
     * @param    string    $url    跳转的地址
     */
    protected function returnTpl($msg = '', $tpl = '', $url = '')
    {
        $tpl = $tpl ?: ($this->request->isMobile() ? 'err' : ROOT_PATH . 'app/v_msg.tpl');
        $this->assign(['msg'=>$msg,'url'=>$url]);
        return $this->fetch($tpl);
    }

    /**
     * @access protected
     * @param  array         $name
     * @param  mixed         $type
     * @return array
     */
    protected function only($name = [], $type = 'post')
    {
        $item = [];
        $data = $this->request->$type(false);
        $preg = [
            'e'=>[2=>'email',3=>'邮箱地址格式错误',4=>'',5=>''],
            'm'=>[2=>'mobile',3=>'手机号码格式错误',4=>'',5=>''],
            'c'=>[2=>'idcard',3=>'身份证号格式错误',4=>'',5=>''],
            'p'=>[2=>'{6,16}',3=>'密码',4=>'5',5=>''],
            'u'=>[2=>'{4,30}',3=>'帐号',4=>'1,2,3',5=>'._@'],
            'n'=>[2=>'{2,30}',3=>'姓名',4=>'0',5=>' .'],
            'i'=>[2=>'{1,30}',3=>'数串',4=>'1',5=>','],
        ];
        foreach($name as $key => $val){
            $default = '';
            $sub = ['','','','','0',' .#-']; // 对应['key','转换类型|验证符*或?','验证规则','提示','合法的字符集','允许的字符']
            if(strpos($val, '/')){
                $sub = explode('/', $val) + $sub;
                $val = $sub[0];
            }
            if(is_int($key)){
                $key = $val;
                if(!key_exists($key,$data) && !$sub[1]){
                    $item[$key] = $default;
                    continue;
                }
            }else{
                $default = $val;
            }
            $v = $data[$key] ?? $default;
            if($sub[1]){
                $must = $msg = true; // $must:是否必须验证  $msg:是否验证不规范时中断反馈提示
                if(in_array($sub[1],['?','$'])){$must = $v ? true : false; if($sub[1] == '$') $msg = false; $sub[1] = '*';}
                switch($sub[1]){
                    case 'a':
                        $v = $v ? (array) $v : [];
                        break;
                    case 'd':
                        $v = (int) $v;
                        break;
                    case 'u':
                        $v = strip_html($v,0);
                        break;
                    case 'h':
                        $v = strip_html($v);
                        break;
                    case 'c':
                        $v = vhtmlspecialchars($v);
                        break;
                    case 'r':
                        $v = dround($v);
                        break;
                    case '*':
                        if($sub[2]=='p') $must = is_md5($v) ? false : $must;
                        $tip = $sub[3]; if(isset($preg[$sub[2]])){$sub = $preg[$sub[2]] + $sub; $tip = $tip ?: $sub[3];}
                        $reg = explode(',',$sub[4]);
                        if($must && !is_preg($v,$sub[2],$reg,$sub[5])){
                            if($msg){
                                $tip = $tip ?: "字段{$key}不合规范"; $txt = ['汉字字母数字下划线','数字','小写字母','大写字母','汉字','任何非空白字符'];
                                if($reg[0]!==''){
                                    $str = ''; foreach($reg as $i){$str .= ($txt[$i] ?? '').'、';}
                                    $tip = $tip.'必须由'.(str_replace(['{',',','}'],['','-',''],$sub[2])).'位'.rtrim($str,'、').($sub[5] ? '和'.str_replace(' ', '空格', $sub[5]) : '').'组成';
                                }
                                $this->exitMsg($tip);
                            }else{
                                $v = '';
                            }
                        }
                        break;
                    case 'f':
                        $v = (float) $v;
                        break;
                    case 'b':
                        $v = (boolean) $v;
                        break;
                    case 's':
                        if(is_scalar($v)){
                            $v = (string) $v;
                        }else{
                            throw new \InvalidArgumentException('variable type error：' . gettype($v));
                        }
                        break;
                }
                if($sub[1] != '*' && $sub[2] && !$v)  $this->exitMsg($sub[2]);
            }
        }
        return $data;
    }
    /**
     * 验证数据
     * @access protected
     *
     * @param array        $data     数据
     * @param string|array $validate 验证器名或者验证规则数组
     * @param array        $message  提示信息
     * @param bool         $batch    是否批量验证
     *
     * @return array|string|true
     * @throws ValidateException
     */
    protected function validate(array $data, $validate, array $message = [], bool $batch = null)
    {
        if (is_array($validate)) {
            $v = new Validate();
            $v->rule($validate);
        } else {
            if (strpos($validate, '.')) {
                // 支持场景
                list($validate, $scene) = explode('.', $validate);
            }
            $class = false !== strpos($validate, '\\') ? $validate : $this->app->parseClass('validate', $validate);
            $v     = new $class();
            if ( ! empty($scene)) {
                $v->scene($scene);
            }
        }

        $v->message($message);

        // 是否批量验证
        if ($batch) {
            $v->batch(true);
        }

        return $v->failException(true)->check($data);
    }
}