<?php
namespace app;

use app\common\traits\HttpTraits;
use think\db\exception\DataNotFoundException;
use think\db\exception\ModelNotFoundException;
use think\Exception;
use think\exception\Handle;
use think\exception\HttpException;
use think\exception\HttpResponseException;
use think\exception\ValidateException;
use think\facade\Log;
use think\Response;
use Throwable;

/**
 * 应用异常处理类
 */
class IndexExceptionHandle extends Handle
{
    use HttpTraits;
    /**
     * 不需要记录信息（日志）的异常类列表
     * @var array
     */
    protected $ignoreReport = [
        HttpException::class,
        HttpResponseException::class,
        ModelNotFoundException::class,
        DataNotFoundException::class,
        ValidateException::class,
    ];

    /**
     * 记录异常信息（包括日志或者其它方式记录）
     *
     * @access public
     * @param  Throwable $exception
     * @return void
     */
    public function report(Throwable $exception): void
    {
        // 使用内置的方式记录异常日志
        parent::report($exception);
    }

    /**
     * Render an exception into an HTTP response.
     *
     * @access public
     * @param \think\Request   $request
     * @param Throwable $e
     * @return Response
     */
    public function render($request, Throwable $e): Response
    {
        // 添加自定义异常处理机制
        if ($e instanceof ValidateException)
        {
            return json(['code' => 500,'msg' => $e->getMessage(),'data' => []]);
        }

        if ($request->post()) {
            Log::info("系统错误" . json_encode([
                    'Message' => $e->getMessage(),
                    'Line' => $e->getLine(),
                    'File' => $e->getFile(),
                    'Trace' => $e->getTraceAsString(),
                ], JSON_UNESCAPED_UNICODE));
            return json(['code' => 500,'msg' => "系统错误",'data' => []]);
        }

        if ($e instanceof HttpException) {
            $code = $e->getStatusCode();
            $msg = $e->getMessage();
            if (!$this->app->isDebug()) {
                return view(
                    $this->app->config->get('app.exception_tmpl'),
                    [
                        'code'    => $code,
                        'message' => $msg,
                        'app_debug' => false
                    ],
                    $code
                );
            }
        }

        // 其他错误交给系统处理
        return parent::render($request, $e);
    }
}
