<?php
namespace app;

use app\common\traits\HttpTraits;
use think\db\exception\DataNotFoundException;
use think\db\exception\ModelNotFoundException;
use think\Exception;
use think\exception\Handle;
use think\exception\HttpException;
use think\exception\HttpResponseException;
use think\exception\ValidateException;
use think\facade\Log;
use think\Response;
use Throwable;

/**
 * 应用异常处理类
 */
class ApiExceptionHandle extends Handle
{
    use HttpTraits;
    /**
     * 不需要记录信息（日志）的异常类列表
     * @var array
     */
    protected $ignoreReport = [
        HttpException::class,
        HttpResponseException::class,
        ModelNotFoundException::class,
        DataNotFoundException::class,
        ValidateException::class,
    ];

    /**
     * 记录异常信息（包括日志或者其它方式记录）
     *
     * @access public
     * @param  Throwable $exception
     * @return void
     */
    public function report(Throwable $exception): void
    {
        // 使用内置的方式记录异常日志
        parent::report($exception);
    }

    /**
     * Render an exception into an HTTP response.
     *
     * @access public
     * @param \think\Request   $request
     * @param Throwable $e
     * @return Response
     */
    public function render($request, Throwable $e): Response
    {
        // 日志记录
        Log::channel('plateform')->error(
            sprintf(
                "请求地址：%s, 请求方式：%s, 请求参数：%s, 错误信息：%s",
                $request->url(),
                $request->method(),
                json_encode($request->param(), JSON_UNESCAPED_UNICODE),
                json_encode([
                    'code' => $e->getCode(),
                    'message' => $e->getMessage(),
                    'line' => $e->getLine(),
                    'file' => $e->getFile(),
                ], JSON_UNESCAPED_UNICODE)
            )
        );
        if ($e instanceof Exception) {
            return $this->fail($e->getMessage());
        }

        return $this->fail("系统错误");
    }
}
